/*
 * %W% %E%
 *
 * @Copyright
 */

package com.sun.jck.lib;

import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import javasoft.sqe.javatest.Status;
import javasoft.sqe.javatest.Test;
import com.sun.javatest.util.WriterStream;

public class DistributedTest implements Test {
    /**
     * Exception used to report internal errors.
     */
    public static class Fault extends Exception {
	/**
	 * Construct a new Fault object that signals failure
	 * with a corresponding message.
	 * 
	 * @param s the string containing a comment
	 */
	public Fault(String s) {
	    super(s);
	}
    }

    /**
     * Create an unnamed distributed test component. The component must
     * be named (see setName) before the internal message client can be
     * created, by run.
     */
    protected DistributedTest() {
    }

    /**
     * Create a distributed test component.
     * @param name  	A name used to identify this component. 
     *			This is used when sending messages between 
     *			components via the message switch.
     */
    protected DistributedTest(String name) {
	this.name = name;
    }
    
    /**
     * The main entry point for the test. The args are decoded via decodeAllArgs,
     * then the message client is initialized and finally, the go() method
     * is called to perform the body of the work.
     */
    public Status run(String[] args, PrintWriter log, PrintWriter ref) {
	this.log = log;
	this.ref = ref;
	try {
	    decodeAllArgs(args);
	    msgClient = new MessageClient(name, msgSwitchAddr) {
		public void handleMessage(String from, String[] args) {
		    DistributedTest.this.handleMessage(from, args);
		}
		public void handleException(Exception e) {
		    DistributedTest.this.handleException(I18n.getString("msg.client.problem"), e);
		}
	    };
	    Status goStatus = go();
	    return (goStatus.getType() >= msgStatus.getType() ?  goStatus : msgStatus);
	}
	catch (IOException e) {
	    printStackTrace(e);
	    return Status.error(I18n.getString("msg.client.cant.start",e));
	}
	catch (DistributedTest.Fault e) {
	    return Status.error(e.getMessage());
	}
    }

    /**
     * The body of the test. This is called from run(...) after the args
     * for the test have been decoded, and the message client has been initialized.
     * The default implementation of this method is to wait until the message
     * client is closed, leaving the class free to respond to incoming messages
     * via the handleMessage mechanism.  The method can be overridden to take
     * proactive control of the test's execution, as might be appropriate for the
     * main (executeClass) component of a DistributedTest.
     */
    protected Status go() {
	msgClient.run();
	return Status.passed("OK");
    }


    /** 
     * Parses the arguments passed to the test.
     *
     * This method embodies the main for loop for all of the 
     * execute arguments. It calls <CODE>decodeArg</CODE> 
     * for successive arguments in the args array.
     *
     * @param args execute arguments from the test harness or from the
     *             command line.
     *
     * @exception Fault raised when an invalid parameter is passed, 
     * or another error occurred.
     *
     * @see #decodeArg 
     */
    protected final void decodeAllArgs( String args[] ) throws Fault {
	/* Please note, we do not increment i
	 * that happens when decodeArg returns the 
	 * number of array elements consumed
	 */
	for (int i = 0 ; i < args.length; ) {
	    int elementsConsumed = decodeArg(args, i);
	    if (elementsConsumed == 0 ) {
		// The argument was not recognized.
		throw new Fault(I18n.getString("recognize.arg.error", args[i]));
	    }
	    i += elementsConsumed;
	}
    }


    /** 
     * Decode the next argument in the argument array.
     * May be overridden to parse additional execute arguments.  
     *
     * @param args execute arguments from the test harness or from the 
     *             command line
     * @param index current index into args.
     *
     * @exception Fault raised when an invalid argument is passed, 
     * or another error occurred.
     *
     * @see #decodeAllArgs 
     */
    protected int decodeArg(String args[], int index) throws Fault {
	if (args[index].equals("-msgSwitch")) {
	    if (index + 1 < args.length) {
		msgSwitchAddr = args[++index];
		return 2;
	    }
	    else 
		throw new Fault(I18n.getString("msgswitch.no.value"));
	}
	else 
	    return 0;
    }

    /**
     * Broadcast a message to all other components of this distributed test.
     */
    public void broadcast(String[] args) throws IOException {
	msgClient.broadcast(args);
    }


    /**
     * Send a message to another component of this distributed test.
     * The component is identified by the name given when the component
     * was created.
     */
    public void send(String name, String[] args) throws IOException {
	msgClient.send(name, args);
    }

    /**
     * Get the name of this component. 
     */
    public String getName() {
	return name;
    }

    /**
     * Set the name of this component. The component must not already be named.
     */
    public void setName(String name) {
	if (name == null)
	    throw new NullPointerException();
	if (this.name != null)
	    throw new IllegalStateException(I18n.getString("already.named.exception"));
	this.name = name;
    }

    public void handleMessages() throws InterruptedException {
	msgClient.start();
    }


    /**
     * Handle a message by invoking a handler method,
     * The name of the method is formed by prefixing the first
     * argument string by the special prefix "handle_".
     * @param from The source of the message
     * @param args The first item identifies a method to be invoked;
     *		subsequent ones are passed the the handler
     */
    void handleMessage(String from, String[] args) {
	try {
	    Method m = getClass().getDeclaredMethod("handle_" + args[0], handleXXXArgs);
	    String[] restOfArgs = new String[args.length - 1];
	    System.arraycopy(args, 1, restOfArgs, 0, restOfArgs.length);
	    Status s = (Status) m.invoke(this, new Object[] { from, restOfArgs });
	    if (s != null)
		setMsgStatus(s);
	}
	catch (IllegalAccessException e) {
	    handleException(I18n.getString("msg.handling.noaccess",args[0]), e);
	}
	catch (InvocationTargetException e) {
	    Throwable et = e.getTargetException();
	    if (et instanceof Fault) 
		setMsgStatus(Status.failed(et.getMessage()));
	    else
		handleException(I18n.getString("msg.handling.exception", args[0]), et);
	}
	catch (NoSuchMethodException e) {
	    handleUnknownMessage(from, args);
	}
    }

    /** 
     * Handle an unknown message (one for which there is no handle_XXX method.)
     * The default is to pront a message and set the message status to error.
     */
    protected void handleUnknownMessage(String from, String[] args) {
	log.println(I18n.getString("msg.unknown",args[0]));
	setMsgStatus(Status.error(I18n.getString("msg.unknown",args[0])));
    }

    /**
     * Print a stack trace for an exception to the log.
     */
    protected void printStackTrace(Throwable t) {
	PrintStream ps = Deprecated.createPrintStream(new WriterStream(log));
	t.printStackTrace(ps);
	ps.close();
    }

    /**
     * Set the message handling status. If the status is already set to
     * the same or a higher level (ERROR > FAILED > PASSED), the new value
     * is ignored.
     */
    private void setMsgStatus(Status s) {
	if (s.getType() > msgStatus.getType())
	    msgStatus = s;
    }

    /**
     * Handle an exception. A message is written to the log, a stack trace is written,
     * and the message status is set to an error.
     */
    private void handleException(String s, Throwable t) {
	log.println(s);
	printStackTrace(t);
	setMsgStatus(Status.error(s));
    }

    private String name;
    protected PrintWriter log;
    protected PrintWriter ref;
    private String msgSwitchAddr;
    private MessageClient msgClient;
    private Status msgStatus = Status.passed("OK");
    private static final Class[] handleXXXArgs = { String.class, String[].class };
}
