/**
 * %W% %E%
 * @Copyright
 */

package	com.sun.javacard.referenceimpl;

import com.sun.javacard.cjck.userinterface.CardProxyException;

import java.io.*;
import java.util.HashSet;
import java.util.Iterator;
import com.sun.javacard.cjck.I18n;

/**
 * A class which manages an exec'd process.
 *
 * @version Version:1.23	Date:11/18/05
 *
 * @author Eric	Nellen (eric.nellen@eng.sun.com)
 */
public class ProcessManager	 {
    public final static long COLLECTOR_SLEEP_TIME = 200;// milliseconds we sleep each iteration
    public final static long CHECKER_SLEEP_TIME = 100;// milliseconds we wait for process to complete
    public final static long SLEEP_TIME = 2000;// milliseconds we sleep each iteration
    public final static int STATE_ERROR = -1;
    private HashSet children = new HashSet();
    private HashSet closed = new HashSet();
    private Logger log;
    private static int checkerCounter = 0;
    private static int collectorCounter = 0;
    
    public class Descriptor {
        private boolean isActive = true;
        public String name;
        public File work_dir;
        public Process process;
        public InputStream out;
        public InputStream err;
        public StringBuffer outBuffer;
        public StringBuffer errBuffer;
        public int state;
	
	String errData;
        
        public Descriptor(String name, String[] prog, String env[], String work_dir) throws CardProxyException {
            this.work_dir = (work_dir == null) ? null : new File(work_dir);
            this.name = name;
            this.outBuffer = new StringBuffer();
            this.errBuffer = new StringBuffer();
            while (true) {
                try {
                    if (log.isTraceEnabled()) {
                        //Convert command elements to string that consists of quoted arguments
                        StringBuffer progCmd = new StringBuffer();
                        for (int i = 0; i < prog.length; i++) {
                            progCmd.append('"').append(prog[i]).append('"').append(' ');
                        }
                        log.log("Trying to start process command: " + progCmd.toString(), Logger.DEBUG_TRACE);
                    }
                    this.process = Runtime.getRuntime().exec(prog, env, this.work_dir);
                    this.err = this.process.getErrorStream();
                    this.out = this.process.getInputStream();
                    //We need to check whether process successfully started
                    //if not log error and throw exception
                    checkProcess(CHECKER_SLEEP_TIME);
                    log.log(I18n.getString("descriptor.name", this, name), Logger.MESSAGE);
                    return;
                } catch (IOException e) {
                    e.printStackTrace();
                    try {
                        Runtime run = Runtime.getRuntime();
                        run.gc();
                        run.runFinalization();
                        Thread.sleep(SLEEP_TIME);	// sleep
                    } catch (InterruptedException ie) {
                        if (this.process != null) {
                            close(true);
                        }
                        state = STATE_ERROR;
                        return;
                    }
                }
            }
        }
        
        /**
         * Checks if the process still alive after given wait time.
         * @param waitTime - time in milliseconds to wait for process completion.
         * @throws CardProxyException if process returns exit value that means
         * process is not alive
         */
        public void checkProcess(long waitTime) throws CardProxyException {
            ProcessChecker checker = new ProcessChecker(this.process);
            checker.start();
            try {
                Thread.sleep(waitTime);
            } catch (InterruptedException ex) {
            }
            if (!checker.isInterrupted()) {
                checker.interrupt();
            }
            if (!checker.isProcessAlive()){
                String errMessage = readAll(err);
		if (errMessage == null || errMessage.length() == 0) {
		    errMessage = errData;
		}
                int exitCode = process.exitValue();
                process.destroy();
                checker = null;
                close(true);
                state = STATE_ERROR;
		log.log("checkProcess() - Process is dead", Logger.DEBUG);
                throw new CardProxyException("Process ends with status '" + exitCode + "'.\nError message: " + errMessage);
            } else {
		log.log("checkProcess() - Process isn't dead", Logger.DEBUG);
                checker = null;
                return;
            }
        }
        
        public int close(boolean isDestroy) {
            if (!isActive) {
                return state;
            }
            try {
                if (isDestroy) {
                    state = this.process.exitValue();
                } else {
                    state = this.process.waitFor();
                }
                return state;
            } catch (InterruptedException ex) {
                throw new RuntimeException(I18n.getString("descriptor.runtime.exception", name));
            } catch (IllegalThreadStateException e) {
                state = STATE_ERROR;
                return state;
            } finally {
                this.process.destroy();
                log.log(I18n.getString("descriptor.name", this.name, (isDestroy ? "' is destroyed" : "' is completed")),
                        Logger.MESSAGE);
                synchronized(this) {
                    readBuffers(this);
                    CJCRECardService.closeStream(this.out);
                    CJCRECardService.closeStream(this.err);
                    isActive = false;
                }
                log.log(I18n.getString("descriptor.closed", this, name), Logger.MESSAGE);
            }
        }
        
        
        /**
         * Thread implementation that checks if a process is alive
         */
        private class ProcessChecker extends Thread {
            
            Process process;
            boolean isProcessAlive = true;
            
            public ProcessChecker(Process process) {
                super("ProcessCheckerThread_" + (++checkerCounter));
                this.process = process;
            }
            
            public void run() {
                try {
                    process.waitFor();
                    isProcessAlive = false;
                } catch (InterruptedException ex) {
                    return;
                }
            }
            
            public boolean isProcessAlive() {
                return isProcessAlive;
            }
        }
    }
    
    
    
    public class Collector extends Thread {
        
        public Collector() {
            super("CollectorThread_" + (++collectorCounter));
        }
        
        private boolean isExecuting = true;
        
        public void run() {
            try {
                while (true) {
                    synchronized (children) {
                        HashSet new_set = new HashSet();
                        for (Iterator e = children.iterator(); e.hasNext();) {
                            Descriptor current = (Descriptor)e.next();
                            (current.isActive ? new_set : closed).add(current);
                            synchronized (current) {
                                readBuffers(current);
                            }
                        }
                        children = new_set;
                    }
                    Thread.sleep(COLLECTOR_SLEEP_TIME);	// sleep
                }
            } catch (InterruptedException e) {
                log.log(I18n.getString("collector.interrupted"), Logger.MESSAGE);
                return;
            }
        }
    }
    
    private Collector collector = new Collector();
    
    public static class Status {
        public String name;
        public int status;
        public String out;
        public String err;
        
        public Status(String name, int status, StringBuffer out, StringBuffer err) {
            this(name, status, out.toString(), err.toString());
        }
        
        public Status(String name, int status, String out, String err) {
            this.name = name;
            this.status = status;
            this.out = out;
            this.err = err;
        }
    }
    /**
     * Creates a new ProcessManager.
     */
    public ProcessManager(Logger log) {
        this.log = log;
    }
    
    public Status execute(String processName, String[] prog, String env[]) {
        Descriptor current = null;
        try {
            current = start(processName, prog, env);
        } finally {
            return waitUntilDone(current);
        }
    }
    
    public Descriptor start(String processName, String[] prog, String env[]) throws CardProxyException {
        return start(processName, prog, env, null);
    }
    
    public Descriptor start(String processName, String[] prog, String env[], String work_dir) throws CardProxyException {
        Descriptor current = new Descriptor(processName, prog, env, work_dir);
        synchronized (children) {
            children.add(current);
        }
        
        if (!collector.isAlive()) {
            collector.start();
        }
        CJCRECardService.closeStream(current.process.getOutputStream());
        return current;
    }
    
    private static Status nullStatus = new Status("null", -1, "", "");
    
    public Status waitUntilDone(Descriptor current) {
        if (current == null) {
            return nullStatus;
        }
        if (current.process == null) {
            return new Status(current.name, -1, "", "");
        }
        
        try {
            return new Status(current.name, current.close(false),
                    current.outBuffer, current.errBuffer);
        } catch (RuntimeException ex) {
            return nullStatus;
        } finally {
            current.close(false);
        }
    }
    
    public void closeAllProcesses() {
        log.log(I18n.getString("start.closeallprocess"), Logger.MESSAGE);
        synchronized (children) {
            for (Iterator e = children.iterator(); e.hasNext();) {
                Descriptor current = ((Descriptor)e.next());
                log.log(I18n.getString("some.process.is.not.closed", current.name),
                        Logger.MESSAGE);
                current.close(true);
            }
            for (Iterator e = closed.iterator(); e.hasNext();) {
                Descriptor current = ((Descriptor)e.next());
                if (current.process != null) {
                    current.process.destroy();
                }
                CJCRECardService.closeStream(current.out);
                CJCRECardService.closeStream(current.err);
            }
        }
        log.log(I18n.getString("finish.closeallprocess"), Logger.MESSAGE);
    }
    
    public String readAll(InputStream in) {
        if (in == null) {
            return "";
        }
        int pos = 0;
        byte[] buff = {};
        try {
            int length = in.available();
	    if (length == 0) {
		return "";
	    }
            buff = new byte[in.available()];
	    int bytesRead = 0;
	    while ((bytesRead = in.read(buff, pos, buff.length - pos)) > 0 && pos < buff.length) {
		pos += bytesRead;
	    }
        } catch (Exception e) {
//            e.printStackTrace();
        }
        return new String(buff, 0 , pos);
    }
    
    private void readBuffers(Descriptor item) {
        if ((item == null) || !item.isActive) {
            return;
        }
        if (item.err != null) {
            String data = readAll(item.err);
	    item.errData = data;
            if (item.errBuffer != null) {
                item.errBuffer.append(data);
            }
        }
        if (item.out != null) {
            String data = readAll(item.out);
            if (item.outBuffer != null) {
                item.outBuffer.append(data);
            }
        }
    }
}
