/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.javacard.cjck.userinterface;
import com.sun.javacard.cjck.I18n;
import com.sun.tck.bvtool.etsi.tlv.TLVUtils;
import com.sun.tck.me.utils.Utils;

public class CommandAPDU extends APDU {
    private byte[] dataIn;
    public static final int MAX_DATA_LENGTH = 32767;
    private int le = -1;
    private int lc = -1;
    public static enum APDU_CASE {
        UNDEFINED(0),
        CASE_1(1), CASE_2S(2), CASE_3S(3), CASE_4S(4),
        CASE_2E(5), CASE_3E(6), CASE_4E(7);

        private APDU_CASE(int order) {
            this.order = order;
        }
        int order;
    }
    private APDU_CASE type;
/*    private static final int CASE_1 = 1;
    private static final int CASE_2S = 2;
    private static final int CASE_3S = 3;
    private static final int CASE_4S = 4;
    private static final int CASE_2E = 5;
    private static final int CASE_3E = 6;
    private static final int CASE_4E = 7;
*/
    /**
     * Creates CommandAPDU with the given header and data bytes. 
     * This constructor doesn't support extended APDU semantics. 
     *
     * @param cla the class byte.
     * @param ins the instruction byte.
     * @param p1 parameter byte.
     * @param p2 parameter byte.
     * @param data the data bytes.
     * @param le the length of the response data.
     */
    public CommandAPDU(byte cla, byte ins, byte p1, byte p2, byte[] data,
                       byte le) {
        this(createData(cla, ins, p1, p2, data, le));
    }

    /**
     * Creates CommandAPDU with the given header and data bytes. 
     * This constructor support both normal and extended APDU semantics. 
     *
     * @param cla the class byte.
     * @param ins the instruction byte.
     * @param p1 parameter byte.
     * @param p2 parameter byte.
     * @param data the data bytes.
     * @param le the length of the response data.
     */
    public CommandAPDU(byte cla, byte ins, byte p1, byte p2, byte[] data,
                       int le) {
        initializeAPDU(cla, ins, p1, p2, data, le);
    }
    
    /**
     * Creates CommandAPDU with the given data bytes.
     * @param data the data bytes.
     */
    public CommandAPDU(byte[] data) {
        super(data);
        calculate();
    }

    public APDU_CASE getCase() {
        return type;
    }
    /**
     * Gets the class byte.
     * @return the class byte.
     */
    public byte getCLA() {
        return data[0];
    }

    /**
     * Gets the instruction byte.
     * @return the instruction byte.
     */
    public byte getINS() {
        return data[1];
    }

    /**
     * Gets the parameter byte.
     * @return the parameter byte.
     */
    public byte getP1() {
        return data[2];
    }

    /**
     * Gets the parameter byte.
     * @return the parameter byte.
     */    
    public byte getP2() {
        return data[3];
    }
    
    /**
     * Gets the lc field.
     * @return the length of the subsequent data bytes.
     */
    public int getLc() {
        return lc;
    }

    /**
     * Gets the le field.
     * @return the length of the response data bytes.
     */
    public int getLe() {
        return le;
    }

    /**
     * Gets the CommandAPDU data byte with the given index.
     * @param index the offset into the data bytes,
     * @return the CommandAPDU data byte at the given index.
     */
    public int getData(int index) {
        if ((index < getLc()) && (index >= 0)) {
            int offset = getDataOffset();
            return (offset < 0) ? -1 :  (data[offset + index] & 0xFF);
        }
        return -1;
    }

    private int getDataOffset() {
        if ((type.order < APDU_CASE.CASE_2E.order) && (type.order > 0)) {
            return 5;
        } else if ((type.order <= APDU_CASE.CASE_4E.order) && (type.order > 0)){
            return 7;
        } else {
            return -1;
        }
    }

    /**
     * Calculates the CommandAPDU bytes.  The CommandAPDU bytes should comply with the ISO7816 standard.
     * <P>Note:<ul>
     * <br><li> In Case 1, both lc and le field are empty.  Consequently, the body of the CommandAPDU is empty.
     * <br><li> In Case 2, lc field is empty, and le field is present.  Consequently, the body of the 
     * CommandAPDU consists of the le field.
     * <br><li> In Case 3, le is empty, and lc is not empty.  Consequently, the body of the CommandAPDU
     * consists of the lc field and followed by the data field.
     * <br><li> In Case 4, both le and lc fields are not empty. Consequently, the body of the CommandAPDU
     * consists of the lc field followed by the data field and le field.
     * </ul>
     */
    public void calculate() {
        if (data.length < 4) {
            return; //ERROR
        }
        if (data.length == 4) { //case 1;
            this.le = 0;
            this.lc = 0;
            this.type = APDU_CASE.CASE_1;
            dataIn = new byte[0];
            return;
        }
        if (data.length == 5) { //case 2S
            this.lc = 0;
            this.le = (data[4] & 0xFF);
            this.type = APDU_CASE.CASE_2S;
            dataIn = new byte[0];
            return;
        }
        
        int b1 = (data[4] & 0xFF);
        if ((b1 != 0) && ((b1 + 5) == data.length)) { //3S
            this.lc = b1;
            this.le = 0;
            this.type = APDU_CASE.CASE_3S;
            dataIn = new byte[this.lc];
            System.arraycopy(data, 5, dataIn, 0, this.lc);
            return;
        } else if ((b1 != 0) && ((b1 + 6) == data.length)) { //4s
            this.lc = b1;
            this.le = data[data.length - 1] & 0xFF;
            this.type = APDU_CASE.CASE_4S;
            dataIn = new byte[this.lc];
            System.arraycopy(data, 5, dataIn, 0, this.lc);
            return;
        }
        
        
        if ((data.length < 7) || (data[4] != 0)) {
            return; // ERROR
        }
        
        int lc = ((data[5] << 8) | (data[6] & 0xFF)) & 0xFFFF;
        if (data.length == 7) { //2E
            this.lc = 0;
            this.le = lc;
            this.type = APDU_CASE.CASE_2E;
            dataIn = new byte[0];
            return;
        }

        if (data.length == (lc + 7)) { //3E
            this.lc = lc;
            this.le = 0;
            this.type = APDU_CASE.CASE_3E;
            dataIn = new byte[this.lc];
            System.arraycopy(data, 7, dataIn, 0, this.lc);
            return;
        }
        
        if (data.length == (lc + 9)) { //4E
            this.lc = lc;
            this.le = (data[data.length - 2] << 8 | (data[data.length - 1] & 0xFF)) & 0xFFFF;
            this.type = APDU_CASE.CASE_4E;
            dataIn = new byte[this.lc];
            System.arraycopy(data, 7, dataIn, 0, this.lc);
        } else {
            return;
        }
    }

    /**
     * Gets the string representation of this CommandAPDU bytes.
     * @return the string representation of this CommandAPDU,
     */
    public String toString() {
        //* following name are ISO standard. so will leave them as is
        StringBuffer buff = new StringBuffer("Command APDU (CLA:");
        buff.append(byteToHex(getCLA()));
        buff.append(", INS:");
        buff.append(byteToHex(getINS()));
        buff.append(", P1:");
        buff.append(byteToHex(getP1()));
        buff.append(", P2:");
        buff.append(byteToHex(getP2()));
        buff.append(", Lc:");
        buff.append(byteToHex(getLc()));
        if (getLc() > 0) {
            buff.append(", Data:0x");
            buff.append(Utils.canonize(data, getDataOffset(), getLc()));
        }
//        for (int i = 0; i < getLc(); i++) {
//          buff.append(", ");
//          buff.append(byteToHex(getData(i)));
//        }
        
        buff.append(", Le:");
        buff.append(byteToHex(getLe()));
        buff.append(")");
        return buff.toString();
    }

     private void initializeAPDU(byte cla, byte ins, byte p1, byte p2, byte[] data, int le) {
        if (data.length > MAX_DATA_LENGTH) {
            throw new IllegalArgumentException(I18n.getString("data.too.long", new Integer(data.length)));
        }
        if (le < 0 || le > MAX_DATA_LENGTH) {
            throw new IllegalArgumentException(I18n.getString("incorrect.le.value", new Integer(le), new Integer(MAX_DATA_LENGTH)));
        }
        this.le = le;
        lc = data.length;
        this.dataIn = new byte[lc];
        if (lc != 0) {
            System.arraycopy(data, 0, dataIn, 0, lc);
        }
        if (lc == 0 && le == 0) {
            type = APDU_CASE.CASE_1;
        } else if (lc == 0) {
            type = (le < 256) ? APDU_CASE.CASE_2S : APDU_CASE.CASE_2E;
        } else if (le == 0) {
            type = (lc < 256) ? APDU_CASE.CASE_3S : APDU_CASE.CASE_3E;
        } else if (le < 256 && lc < 256) {
            type = APDU_CASE.CASE_4S;
        } else {
            type = APDU_CASE.CASE_4E;
        }
        this.data = createAPDUByType(cla, ins, p1, p2, data, le, type);
    }

     /**
     * Creates CommandAPDU with the given header and data. This
     * constructor does not support extended APDUs.
     * @param cla the class byte.
     * @param ins the instruction byte.
     * @param p1 parameter byte.
     * @param p2 parameter byte.
     * @param data the data bytes.
     * @param le the length of the response data
     * @return byte array containing the CommandAPDU bytes.
     */
    public static byte[] createData(byte cla, byte ins, byte p1, byte p2,
                                    byte[] data, byte le) {
        if (data.length > 255) {
            throw new IllegalArgumentException(I18n.getString("data.too.long", new Integer(data.length)));
        }
        int length = (4 + data.length + ((data.length == 0) ? 0 : 1) +
                      ((le == 0) ? 0 : 1));
        byte[] retVal = new byte[length];
        retVal[0] = cla;
        retVal[1] = ins;
        retVal[2] = p1;
        retVal[3] = p2;
        if (data.length != 0) {
            retVal[4] = (byte)data.length;
            System.arraycopy(data, 0, retVal, 5, data.length);
        }
        if (le != 0) {
            retVal[retVal.length - 1] = le;
        }
        return retVal;
    }
    /**
     * This method returns byte array corresponding to APDU
     * with supplied parameters.
     *
     * @param cla the class byte.
     * @param ins the instruction byte.
     * @param p1 parameter byte.
     * @param p2 parameter byte.
     * @param data the data bytes.
     * @param le the length of the response data.
     * @param type APDU type (case) according to ISO7816-3
     *
     * @return byte array corresponding to APDU
     */
    private static byte[] createAPDUByType(byte cla, byte ins, byte p1, byte p2,
            byte[] data, int le, APDU_CASE type) {
        byte[] apdu = null;
        int lcLength = 0;
        switch(type) {
            case CASE_1:
                return new byte[]{cla, ins, p1, p2};
            case CASE_2S:
                return new byte[]{cla, ins, p1, p2, (byte)le};
            case CASE_2E:
                return new byte[]{cla, ins, p1, p2, (byte)0, (byte)(le >> 8), (byte)(le & 0x00FF)};
            case CASE_3S:
                apdu = new byte[5 + data.length];
                apdu[4] = (byte)data.length;
                lcLength = 1;
                break;
            case CASE_3E:
                apdu = new byte[7 + data.length];
                apdu[4] = (byte)0;
                apdu[5] = (byte)(data.length >> 8);
                apdu[6] = (byte)(data.length & 0x00FF);
                lcLength = 3;
                break;
            case CASE_4S:
                apdu = new byte[6 + data.length];
                apdu[4] = (byte)data.length;
                apdu[5 + data.length] = (byte)le;
                lcLength = 1;
                break;
            case CASE_4E:
                apdu = new byte[9 + data.length];
                apdu[4] = (byte)0;
                apdu[5] = (byte)(data.length >> 8);
                apdu[6] = (byte)(data.length & 0x00FF);
                apdu[7 + data.length] = (byte)(le >> 8);
                apdu[8 + data.length] = (byte)(le & 0x00FF);
                lcLength = 3;
                break;
        }
        apdu[0] = cla;
        apdu[1] = ins;
        apdu[2] = p1;
        apdu[3] = p2;
        System.arraycopy(data, 0, apdu, 4 + lcLength, data.length);
        return apdu;
    }

    public byte[] getDataIn() {
        if (dataIn == null) {
            return null;
        }
        byte[] result = new byte[dataIn.length];
        System.arraycopy(dataIn, 0, result, 0, dataIn.length);
        return result;
    }
}
