/*
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.userinterface;

import com.sun.tck.bvtool.terminal.StatefulCardTerminal;
import java.io.PrintWriter;
import java.util.Hashtable;

/**
 * @author Eric Nellen (eric.nellen@eng.sun.com), Dmitri Trounine
 */
public interface CardService {
    
    /**
     *  Constant for the contacted i/o interface. To be used as a parameter for
     *  the <code>getCommunicationService(int iface)</code> method.
     */
    public static final int CONTACTED_INTERFACE =  1;
    
    /**
     *  Constant for the contactless i/o interface. To be used as a parameter for
     *  the <code>getCommunicationService(int iface)</code> method.
     */
    public static final int CONTACTLESS_INTERFACE = 2;
    
    /**
     * This named constant is used as a key in the Hashtable returned by
     * {@link #getProperties() getProperties()}.
     * <br>It is a key for the licensee's Proxy name.
     * <br>Note:<ul>This value is used for information reporting purposes only.</ul>
     */
    public static final String PROXY_NAME_PROPERTY        = "Proxy Name";
    
    /**
     * This named constant is used as a key in the Hashtable returned by
     * {@link #getProperties() getProperties()}.
     * <br>It is a key for the licensee's Proxy version ID string.
     * <br>Note:<ul>This value is used for information reporting purposes only.</ul>
     */
    public static final String PROXY_VERSION_PROPERTY        = "Proxy Version";
    
    /**
     * This named constant is used as a key in the Hashtable returned by
     * {@link #getProperties() getProperties()}.
     * <br>It is a key for the licensee's Proxy version ID string.
     * <br>The value in the Hashtable is a String.
     * <br>Note:<ul>This value is used for information reporting purposes only.</ul>
     */
    public static final String API_VERSION_PROPERTY = "API Version";
    
    /**
     * This named constant is used as a key in the Hashtable returned by
     * {@link #getProperties() getProperties()}.
     * <br>It is a key for the maximum stack size in bytes of the licensee's Proxy.
     * <br>The value in the Hashtable is an Integer.
     * <br>Note:<ul>This value may be used for computation by the cJCK.</ul>
     */
    public static final String MAX_STACK_PROPERTY = "Max Stack";
    
    /**
     * This named constant is used as a key in the Hashtable returned by
     * {@link #getProperties() getProperties()}.
     * <br>It is a key for the RAM size in bytes of the licensee's Proxy,
     * which is available for transient data.
     * <br>The value in the Hashtable is an Integer.
     * <br>Note:<ul>This value may be used for computation by the cJCK.</ul>
     */
    public static final String MAX_TRANSIENT_RAM_PROPERTY = "Max Transient RAM";
    
    /**
     * This named constant is used as a key in the Hashtable returned by
     * {@link #getProperties() getProperties()}.
     * <br>It is a key for the maximum size in bytes of non-volatile RAM
     * supported by the licensee's Proxy.
     * <br>The value in the Hashtable is an Integer.
     * <br>Note:<ul>This value may be used for computation by the cJCK.</ul>
     */
    public static final String MAX_NVRAM_PROPERTY = "Max NVRAM";
    
    /**
     * This named constant is used as a key in the Hashtable returned by
     * {@link #getProperties() getProperties()}.
     * <br>It is a key for the attribute which indicates whether
     * the licensee's Proxy provides support for ints or not.
     * <br>The value in the Hashtable is a Boolean.
     * <br>Note:<ul>This value may be used for computation by the cJCK.</ul>
     */
    public static final String INT_SUPPORT_PROPERTY = "int support";
    
    /**
     * Prepare to start a test.  Initialize the Proxy.
     * Called by the cJCK to notify the Card Service that a test is starting, and
     * specify the path of the working directory.
     * <p>Note:<ul>
     * <li>The Card Service should prepare the working directory for the test.
     * This includes deleting any intermediate files that may have been left over
     * from a prior test execution.
     * The cJCK assumes that no test applets exist in either the ROM image or EEPROM.
     * </ul>
     * @param workingDir
     *        The <tt>working directory</tt> to be used for this test.
     * @param numberOfExecutions
     *        Indicates the number of times that the proxy will be powered on.
     * @exception CardProxyException
     *        Thrown if unable to initialize or set the working directory.
     * @see #stopTest()
     */
    public void startTest(String workingDir, int numberOfExecutions)
    throws CardProxyException;
    
    
    /**
     * Stop the test.  Terminate the Proxy.
     * Called by the cJCK to notify the Card Service that a test is stopping.
     * @see CJCKCardService#startTest
     * @exception CardProxyException
     *     Thrown if unable to terminate the Proxy.
     */
    public void stopTest() throws CardProxyException;
    
    /////////////////////////////////////////////////////////////////////////////
    // Utility functions
    /////////////////////////////////////////////////////////////////////////////
    
    /**
     * Converts the specified applet and creates a CAP file.
     * <P>This method enables the cJCK to invoke a Converter.
     * <br>The goal is to verify the functionality of the Converter.
     * <P>The cJCK will attempt to convert both "correct" and "incorrect" applets.<ul>
     * <li>An "incorrect" applet is one which violates the Java Card Language subset.
     * <li>A "correct" applet is one which conforms to the Java Card Language subset
     * and which runs correctly on the Reference Implementation.
     * The resulting CAP file will be installed and the test applet exercised to verify that
     * conversion was done correctly.  The byte code sequences in the CAP file are not examined.
     * </ul>
     * <br>How the conversion is accomplished is at the discretion of the implementer.
     * <P>Implementation Notes:<ul>
     * <li>When implementing this method, you should allow your Converter to
     * write messages to Standard Out and Standard Error.  You may want to
     * invoke your Converter with a "verbose" option to provide as much diagnostic
     * information as possible.
     * All output is captured in each test's .jtr file and can be viewed with a file editor
     * or from within JavaTest.
     * </ul>
     * @param packageName the name of the package being converted.
     * @param packageAid the AID of the package being converted.
     * @param majorVersion the major version of the package.
     * @param minorVersion the minor version of the package.
     * @param classRootDir the root directory of the class hierarchy.
     * The path /java/lang/javacard is appended to this root,
     * yielding the directory in which the Converter will look for classes.
     * @param exportRootDir the root directory of the path in which the
     * Converter will search for export files.
     * @param isExportMap if it is true, then the package should be
     * converted using the <code>export</code> file of the preexisting
     * cap file. The export file should be searched in the
     * <code>exportRootDir</code>.
     * @param outputRootDir the root directory for output.
     * The Export File, if specified, must be put into the directory at
     * the outputRootDir and the path implied by the package name.
     * Any intermediate or other files produced by the Converter
     * must also go into this directory.
     * @param apa[] an array of {@link com.sun.javacard.cjck.userinterface.AppletProperties}
     * which specifies the applets to be converted.
     * Each appletProperties entry must include name of class that defines the install method.
     * @return the fully-qualified file name of the cap file that was created.
     * @exception CardProxyException
     * if the Converter is unable to create the CAP file.
     * The exception message should be appropriately descriptive.
     * @see com.sun.javacard.cjck.userinterface.AppletProperties
     */
    public String convertPackage(String packageName,
            AppletID packageAid,
            int majorVersion,
            int minorVersion,
            String classRootDir,
            String exportRootDir,
            boolean isExportMap,
            String outputRootDir,
            AppletProperties[] apa)
            throws CardProxyException;
    
    
    /////////////////////////////////////////////////////////////////////////////
    // Loading functions
    /////////////////////////////////////////////////////////////////////////////
    
    
    /**
     * Loads the specified applets from their class files into the Proxy.
     * This method should be implemented for pre-issuance Products, which
     * support class file format.
     * <br>How the loading is done and where the applets are loaded is at the discretion
     * of the implementer.
     * <P>Note:<ul>
     * <li><em>Only the applet's package name, class name, AID, and params
     * are guaranteed to be valid.</em>
     * <li>Values of other applet properties, such as MAX_STACK, will be -1 if not specified.
     * <li>The implementer should use default values whenever a value of -1 is encountered.
     * </ul>
     * @param packageNames an array with the package names. The array contains names of the all packages,
     * which are required for the test execution.
     * @param appletProperties an {@link AppletProperties} array which specifies the applets to be loaded.
     * @param classRootDir the root directory of the class hierarchy.
     * The path /java/lang/javacard is appended to this root,
     * yielding the directory in which the Converter will look for classes.
     * @param outputRootDir the root directory for output.
     * The Export File, if specified, must be put into the directory at
     * the outputRootDir and the path implied by the package name.
     * Any intermediate or other files produced by the Converter
     * must also go into this directory.
     * @return true if the packages with applets are installed
     * successfully, false otherwise. If the applet's install method
     * throws an exception, then loadClassFileApplets should return false
     * instead of throwing CardProxyException.
     * @exception CardProxyException
     *     if the Proxy is unable to load an applet in the list.
     *     The exception message should be appropriately descriptive.
     */
    public boolean loadClassFileApplets(String[] packageNames,
            AppletProperties [] appletProperties,
            String classRootDir,
            String outputRootDir) throws CardProxyException;
    
    /**
     * Loads the specified applets from their cap files into the Proxy.
     * This method should be implemented for pre-issuance Products, which
     * support CAP file format.
     * <br>How the loading is done and where the applets are loaded
     * is at the descriptive of the implementer.
     * <P>Note:<ul>
     * <li><em>Only the applet's package name, class name, AID, and params
     * are guaranteed to be valid.</em>
     * <li>Values of other applet properties, such as MAX_STACK, will be -1 if not specified.
     * <li>The implementer should use default values whenever a value of -1 is encountered.
     * </ul>
     * @param capfiles array with the cap file names. The array contains names
     * of the all cap files, which are required for the test execution.
     * @param appletProperties an {@link AppletProperties} array
     *     which specifies the applets to be loaded.
     * @param outputRootDir the root directory for output.
     * The Export File, if specified, must be put into the directory at
     * the outputRootDir and the path implied by the package name.
     * Any intermediate or other files produced by the Converter
     * must also go into this directory.
     * @return true if the package with applets are installed
     * successfully, false otherwise. If the applet's install method
     * throws an exception, then loadCapFileApplets should return false
     * instead of throwing CardProxyException.
     * @exception CardProxyException
     *     if the Proxy is unable to load an applet in the list.
     *     The exception message should be appropriately descriptive.
     */
    public boolean loadCapFileApplets(String[] capfiles,
            AppletProperties [] appletProperties,
            String outputRootDir)
            throws CardProxyException;
    
    
    
    /**
     * Installs a CAP file into the Proxy. This method should be implemented for
     * post-issuance products.
     * <P>Note:<ul>
     * Assumes that powerUp() has already been called.
     * </ul>
     * @param appletPropertiesArray an {@link AppletProperties} array which
     *     specifies the applets in the CAP file. If the cap file contains
     *     library package, then the array has zero length.
     * @param aCAPFileName is a fully-qualified file name including the path name.
     * @param outputRootDir the root directory for output.
     * Any intermediate or other files produced by the Converter must be created in
     * the directory at the outputRootDir and the path implied by the package name.
     * @return true if the package with applets are installed
     * successfully, false otherwise. If the applet's install method
     * throws an exception, then installCAPFile should return false
     * instead of throwing CardProxyException.
     * @exception CardProxyException
     *     thrown if unrecoverable error occurs during the cap file installation
     *     (connection is interrupted, implementation exits abruptly).
     *     The exception message should be appropriately detailed and descriptive.
     *     <br>
     *     Note that if the installer returns error status APDU, then
     *     CardProxyException should not be thrown and the method should
     *     return false.
     */
    public boolean installCAPFile(AppletProperties[] appletPropertiesArray,
            String aCAPFileName,
            String outputDir)
            throws CardProxyException;
    
    public void setCardTerminal(StatefulCardTerminal terminal);

    /**
     * deletes list of the applets.
     * @param applets specifies list of the applets for deletion.
     * @return true if the all applets are deleted successfully, false otherwise.
     * @exception CardProxyException
     *     thrown if unable to delete applet instances.
     */
    public boolean deleteAppletInstances(AppletProperties[] applets)
    throws CardProxyException;
    
    /**
     * deletes given applet/library package.
     * @param packageID specifies AID of the given package.
     * @param packageName specifies full qualified name of the given package.
     * @return true if the package are deleted successfully, false otherwise.
     * @exception CardProxyException
     *     thrown if unable to delete package.
     */
    public boolean deletePackage(AppletID packageID, String packageName)
    throws CardProxyException;
    
    /**
     * deletes given applet/library package and contained instances.
     * @param packageID specifies AID of the given package.
     * @param packageName specifies full qualified name of the given package.
     * @return true if the package are deleted successfully, false otherwise.
     * @exception CardProxyException
     *     thrown if unable to delete package and contained instances.
     */
    public boolean deletePackageAndInstances(AppletID packageID,
            String packageName)
            throws CardProxyException;
    
    /**
     * initializes the CardService instance. Test Run Framework invokes this
     * method before invocation of any other methods of the interface.
     * @param args specifies initialization options.
     * These options are defined in the jte file by option -s.
     * @param out specifies PrintWriter for output.
     * @param ref a stream to which to write any diagnostic messages.
     */
    public void init(String[] args, PrintWriter out, PrintWriter ref);
    
    
    /**
     * Retrieve the Proxy properties Hashtable.
     * The cJCK will call this method to retrieve the properties of the Proxy.
     * <br>The implementor is required to initialize the Hashtable with the keys listed below
     * and their associated values.
     * <br>Note that some values are String values, some are Integers, and some are booleans.
     * @return Hashtable containing the properties associated with the Proxy Properties constants.
     * <br>Valid keys are the static constants:
     * {@link #PROXY_VERSION_PROPERTY PROXY_VERSION_PROPERTY},
     * {@link #API_VERSION_PROPERTY API_VERSION_PROPERTY},
     * {@link #MAX_STACK_PROPERTY MAX_STACK_PROPERTY},
     * {@link #MAX_TRANSIENT_RAM_PROPERTY MAX_TRANSIENT_RAM_PROPERTY},
     * {@link #MAX_NVRAM_PROPERTY MAX_NVRAM_PROPERTY},
     * {@link #INT_SUPPORT_PROPERTY INT_SUPPORT_PROPERTY}
     */
    public Hashtable getProperties();
    
    /**
     *  Get a CommunicationService instance for specified i/o interface.
     *
     *  @param  iface   The I/O interface, that the requested CommunicationService will work over.
     *
     *  @exception  CardProxyException  if requested i/o interface is not avaliable.
     */
    public CommunicationService getCommunicationService(int iface) throws CardProxyException;
}
