/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.scripts.rmi;

import com.sun.javacard.cjck.scripts.ScriptFailException;
import com.sun.javacard.cjck.scripts.RMIScript;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.I18n;

public class Parameter {

    protected byte[] bytes;
    protected String name;

    protected Parameter() {
    }

    public static int makeInt(byte b1, byte b2, byte b3, byte b4) {
        return (((b1 << 24) & 0xFF000000) | ((b2 << 16) & 0xFF0000)
                | ((b3 << 8)  & 0xFF00) | (b4 & 0xFF));
    }

    public static Parameter parseReturnData(byte[] data, int expected_type)
        throws ResponseException, CardProxyException {
        Parameter retVal = null;
        if (expected_type == RMIScript.VOID) {
            RMIScript.check((data == null) || (data.length == 0),
                            I18n.getString("void.should.return.zero"));
            return new Void();
        }
        if ((expected_type & 0xf0) == RMIScript.ARRAY) {
            if ((data.length == 1) && (data[0] == RMIScript.NULL_TAG)) {
                throw new ScriptFailException(I18n.getString("correct.remote.null.ref.should.be.used"));
            }
            if ((data.length == 2)
                && (data[0] == RMIScript.NULL_TAG)
                && (data[1] == RMIScript.NULL_TAG)) {
                return new Null();
            }
        }
        switch (expected_type) {
        case RMIScript.BOOLEAN:       retVal = new Boolean(false); break;
        case RMIScript.BYTE:          retVal = new Byte((byte)0); break;
        case RMIScript.SHORT:         retVal = new Short((short)0); break;
        case RMIScript.INT:           retVal = new Int(0); break;
        case RMIScript.BOOLEAN_ARRAY: retVal = new BooleanArray(null); break;
        case RMIScript.BYTE_ARRAY:    retVal = new ByteArray(null); break;
        case RMIScript.SHORT_ARRAY:   retVal = new ShortArray(null); break;
        case RMIScript.INT_ARRAY:     retVal = new IntArray(null); break;
        }
        retVal.bytes = data;
        return retVal;
    }
    public byte[] getBytes() {
        return bytes;
    }
    
    public String getType() {
        return name;
    }

    public boolean isNull() {
        return false;
    }
    
    public static class Void extends Parameter {
        public Void() {
            this.name = I18n.getString("void.parameter");
            this.bytes = new byte[0];
        }
    }
        
    public static class Raw extends Parameter {
        public Raw(byte[] data) {
            this.name = I18n.getString("raw.parameter");
            this.bytes = data;
        }
    }
    
    public static class Boolean extends Parameter {
        
        public Boolean(boolean value) {
            this((byte)(value ? 1 : 0));
        }

        public Boolean(byte value) {
            this.name = I18n.getString("boolean.parameter");
            this.bytes = new byte[] {value};
        }
        public boolean getValue() {
            return (this.bytes[0] != 0);
        }
    };

    public static class Byte extends Parameter {
        
        public Byte(byte value) {
            this.name = I18n.getString("byte.parameter");
            this.bytes = new byte[] {value};
        }
        public byte getValue() {
            return this.bytes[0];
        }
    };
    
    public static class Short extends Parameter {
        public Short(short value) {
            this.name = I18n.getString("short.parameter");
            this.bytes = new byte[] {
                (byte)(value >> 8),
                (byte)value
            };
        }
        
        public short getValue() {
            return (short)Parameter.makeInt((byte)0, (byte)0, this.bytes[0],
                                            this.bytes[1]);
        }
    }
    
    public static class Int extends Parameter {
        public Int(int value) {
            this.name = I18n.getString("integer.parameter");
            this.bytes = new byte[] {
                (byte)(value >> 24),
                (byte)(value >> 16),
                (byte)(value >> 8),
                (byte)value
            };
        }
        public int getValue() {
            return Parameter.makeInt(this.bytes[0], this.bytes[1],
                                     this.bytes[2], this.bytes[3]);
        }
    }

    public static class Null extends Parameter {
        public Null() {
            this.name = I18n.getString("null.array.parameter");
            this.bytes = new byte[] { (byte)0xFF };
        }
        public boolean isNull() {
            return (this.bytes == null) ||
                (this.bytes.length == 1) && (this.bytes[0] == (byte)0xFF);
        }

    };

    
    public static class BooleanArray extends Null {
        public BooleanArray(boolean[] value) {
            this.name = I18n.getString("boolean.array.parameter");
            if (value == null) {
                this.name = I18n.getString("null.boolean.array.parameter");
                this.bytes = new byte[] { (byte)0xFF };
            } else {
                this.name = I18n.getString("boolean.array.parameter");
                this.bytes = new byte[value.length + 1];
                this.bytes[0] = (byte)value.length;
                for (int i = 0; i < value.length; i++) {
                    this.bytes[1 + i] = (byte)(value[i] ? 1 : 0);
                }
            }
        }
        public boolean[] getValue() {
            checkLength(this.bytes, 1);
            boolean[] retVal = new boolean[this.bytes[0]];
            for (int i = 0; i < retVal.length; i++) {
                retVal[i] = (this.bytes[i + 1] != 0);
            }
            return retVal;
        }
    };
    
    public static class ByteArray extends Null {
        public ByteArray(byte[] value) {
            if (value == null) {
                this.name = I18n.getString("null.byte.array.parameter");
                this.bytes = new byte[] { (byte)0xFF };
            } else {
                this.name = I18n.getString("byte.array.parameter");
                this.bytes = new byte[value.length + 1];
                this.bytes[0] = (byte)value.length;
                System.arraycopy(value, 0, this.bytes, 1, value.length);
            }
        }
        public byte[] getValue() {
            checkLength(this.bytes, 1);
            byte[] retVal = new byte[this.bytes[0]];
            System.arraycopy(this.bytes, 1, retVal, 0, retVal.length);
            return retVal;
        }
    };
       
    public static class ShortArray extends Null {
        public ShortArray(short[] value) {
            if (value == null) {
                this.name = I18n.getString("null.short.array.parameter");
                this.bytes = new byte[] { (byte)0xFF };
            } else {
                this.name = I18n.getString("short.array.parameter");
                this.bytes = new byte[value.length * 2 + 1];
                this.bytes[0] = (byte)value.length;
                for (int i = 0; i < value.length; i++) {
                    this.bytes[1 + i * 2] = (byte)(value[i] >> 8);
                    this.bytes[1 + i * 2 + 1] = (byte)value[i];
                }
            }
        }
        public short[] getValue() {
            checkLength(this.bytes, 2);
            short[] retVal = new short[this.bytes[0]];
            for (int i = 0; i < retVal.length; i++) {
                retVal[i] = (short)Parameter.makeInt((byte)0, (byte)0,
                                                     this.bytes[i*2 + 1],
                                                     this.bytes[i * 2 + 2]);
            }
            return retVal;
        }
    };

    public static class IntArray extends Null {
        public IntArray(int[] value) {
            if (value == null) {
                this.name = I18n.getString("null.int.array.parameter");
                this.bytes = new byte[] { (byte)0xFF };
            } else {
                this.name = I18n.getString("int.array.parameter");
                this.bytes = new byte[value.length * 4 + 1];
                this.bytes[0] = (byte)value.length;
                for (int i = 0; i < value.length; i++) {
                    this.bytes[1 + i * 4] = (byte)(value[i] >> 24);
                    this.bytes[1 + i * 4 + 1] = (byte)(value[i] >> 16);
                    this.bytes[1 + i * 4 + 2] = (byte)(value[i] >> 8);
                    this.bytes[1 + i * 4 + 3] = (byte)value[i];
                }
            }
        }
        
        public int[] getValue() {
            checkLength(this.bytes, 4);
            int[] retVal = new int[this.bytes[0]];
            for (int i = 0; i < retVal.length; i++) {
                retVal[i] = Parameter.makeInt(this.bytes[i*4 + 1],
                                              this.bytes[i*4 + 2],
                                              this.bytes[i*4 + 3],
                                              this.bytes[i*4 + 4]);
            }
            return retVal;
        }
    };
    
    public static void checkLength(byte[] data, int TYPE_SIZE)
        throws ScriptFailException {
        if ((data == null) || (data.length == 0)) {
            throw new ScriptFailException(I18n.getString("array.is.null.or.has.zero.length"));
        }
        int expected_length = (data[0] & 0xFF) * TYPE_SIZE + 1;
        if (expected_length != data.length) {
            throw new ScriptFailException(I18n.getString("array.has.incorrect.length", 
                                                         Integer.toHexString(expected_length), 
                                                         Integer.toHexString(data.length)));
        }
    }
}
