/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.invoke;

import com.sun.javacard.cjck.userinterface.AppletProperties;
import com.sun.javacard.cjck.userinterface.AppletID;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.Date;
import com.sun.javacard.cjck.I18n;


/**
 * This class reads a configuration file and 
 * interprets information about applet classes.
 * This class is used by the {@link com.sun.javacard.referenceimpl.JCWDECardService};
 * <em>it is not part of the {@link com.sun.javacard.cjck.userinterface.CJCKCardService} interface.</em>
 * A ConfigFileReader can return the applet information as either 
 * an array {@link #getAppletPropertiesArray()},
 * a vector {@link #getAppletVector()},
 * an enumeration {@link #elements()},
 * or a string {@link #toString}.
 *
 * A config file contains one applet per non-comment line in the following format:
 * <br><code><pre>
 * <br>// applet                                 AID
 * <br>package com.sun.javacard.JavaLoyalty      0x4C:6F:79:61:6C:74:01
 * <br>com.sun.javacard.JavaLoyalty.JavaLoyalty  0x4C,6F,79,61,6C,74,79
 * <br>com.sun.javacard.JavaLoyalty.JavaLoyalty  0x4C:6F:79:61:6C:74:79
 * </pre>
 *
 * @version Version:1.17 Date:05/16/05
 *
 * @author Eric Nellen (eric.nellen@eng.sun.com)
 */

public class ConfigFileReader {

    static String BUILD_VERSION = "Version: %Z%";
    static String BUILD_DATE = "%E%";

    private static PrintWriter stdOut;
    private static PrintWriter stdErr;
    
    private static final int MAX_APPS = 100;    // the upper limit
    private final static String EOL = System.getProperty("line.separator", "\n");

    static byte PACKAGE_AID_LAST_BYTE_MASK = 0xF;
    
    String configFile;
    private Vector appVec;
    private Vector packageOrder; 
    private Hashtable packages;
    static String config_id;
    

    /**
     * Constructor.
     */
    public ConfigFileReader(PrintWriter out, PrintWriter err) {
        appVec = new Vector();
        packages = new Hashtable();
        packageOrder = new Vector();
        stdOut = (out == null) ? new PrintWriter(System.out, true) : out;
        stdErr = (err == null) ? new PrintWriter(System.err, true) : err;
    }
        
    /**
     * This method reads a configuration file from which it extracts
     * applet name, AID, and installation parameter info.
     * @param file the file which contains applet information.
     */
    public void read(String file)
        throws java.io.IOException {

        config_id=I18n.getString("config.id");
        log(I18n.getString("config.verion", config_id, BUILD_VERSION, BUILD_DATE));
        configFile = file;
        String line;
        String aidString;
        String packageName;
        AppletID packageAID;
        AppletID aid;
        byte[] aidBytes;
        byte[] packageBytes;
        AppletProperties appProperties;
        
        BufferedReader in = new BufferedReader(new FileReader(file));
        try {
            while ((line = in.readLine())!= null) {

                // Trim and replace tabs by spaces
                line = line.trim().replace('\t', ' ');

                // Skip comments or blank lines
                if (line.startsWith("//") || (line.length()==0))
                    continue;

                StringTokenizer tokens = new StringTokenizer(line, " ");

                if (tokens.countTokens() < 2) {
                    throw new RuntimeException(I18n.getString("missing.aid", line));
                }
            
                // first token is assumed to be class name or package
                // keyword for library packages
                String fullname = tokens.nextToken();

                if (fullname.equals("package")) {
                    parsePackage(tokens);
                    continue;
                }
                packageName = fullname.substring(0, fullname.lastIndexOf("."));
            
                // example:
                // com.sun.javacard.cjck.tests.vm.instr._aaload.aaload001.aaload001 "0xA0:0x00:0x00:0x00:0x62:0x03:0x01:0x0A:0x77:0x24:0x81"
            
                // second token is assumed to be AID
                aidString = tokens.nextToken();
                aidBytes = convertHexToBytes(aidString, AppletProperties.MAX_AID_LENGTH + 10);

                packageAID = getPackageAID(packageName);

                if (packageAID == null) {
                    packageBytes = new byte[aidBytes.length];
                    java.lang.System.arraycopy(aidBytes, 0, packageBytes, 0,
                                               aidBytes.length);
                    packageBytes[packageBytes.length - 1] &= PACKAGE_AID_LAST_BYTE_MASK;
                    packageAID = new AppletID(packageBytes);
                }
            
                aid = new AppletID(aidBytes);


                appProperties = new AppletProperties(packageName, packageAID, fullname);
                appProperties.setAID(aid);

                addPackage(packageName,  packageAID,
                           (tokens.hasMoreTokens() ? tokens.nextToken() : null),
                           tokens.hasMoreTokens());
                addApplet(packageName, packageAID, appProperties);
            }
        } finally {
            try {
                if (in != null) {
                    in.close();
                }
            } catch (IOException e) {
            }
        }
    }
    
    private void parsePackage(Enumeration e) {
        String name = (String)e.nextElement();
        byte[] aidBytes = convertHexToBytes((String)e.nextElement(),
                                            AppletProperties.MAX_AID_LENGTH + 10);
        String version = e.hasMoreElements() ? (String)e.nextElement() : null;
        addPackage(name, new AppletID(aidBytes), version, e.hasMoreElements());
    }
       
    private void addPackage(String name, AppletID aid, String version,
                            boolean exportMap) {
        Info last = (Info)packages.get(name);
        if (last == null) {
            packages.put(name, new Info(name, aid, version, exportMap));
            packageOrder.addElement(name);
        } else {
            if (!aid.equals(last.getPackageAID()) ||
                ((version != last.getVersion()) && (version != null))) {
                // The versions are interned and can be compared by == operator
                throw new IllegalArgumentException(I18n.getString("package.aid.not.equal"));
            }
        }
    }
    
        
    private void addApplet(String name, AppletID aid, AppletProperties ap) {
        Info last = (Info)packages.get(name);
        last.addApplet(ap);
        appVec.addElement(ap);
    }

    public Enumeration getPackages() {
        return packageOrder.elements();
    }

    public AppletID getPackageAID(String name) {
        Info data = (Info)packages.get(name);
        return (data == null) ? null : data.getPackageAID();
    }
    
    public boolean isExportMap(String name) {
        Info data = (Info)packages.get(name);
        return data.isExportMap();
    }
    
    public String getPackageVersion(String name) {
        Info data = (Info)packages.get(name);
        return (data == null) ? null : data.getVersion();
    }
    
    public AppletProperties[] getApplets(String name) {
        Info data = (Info)packages.get(name);
        return (data == null) ? null : data.getApplets();
    }
    
    public AppletProperties[] getPropertiesByCapFile(String name)
        throws IllegalArgumentException {
        String shortName = name;
        if (name.endsWith(".cap"))
            shortName = name.substring(0, name.lastIndexOf(".cap"));
        shortName = ArgumentParser.getBasename(shortName);
        Vector h = new Vector();
        for (Enumeration e = getPackages(); e.hasMoreElements();) {
            String next = (String)e.nextElement();
            String s =  next.replace('.', File.separatorChar);
            s = ArgumentParser.getBasename(s);
            if (shortName.equals(s))
                h.addElement(next);
        }
        if (h.size() == 0)
            throw new IllegalArgumentException(I18n.getString("package.not.found", name));
        if (h.size() != 1)
            throw new IllegalArgumentException(I18n.getString("more.packages.found", name));
        return getApplets((String)h.elementAt(0));
    }
       
    /**
     * Return an Array of the applets
     */
    public AppletProperties [] getAppletPropertiesArray() {
        return (AppletProperties[])appVec.toArray(new AppletProperties[appVec.size()]);
    }

    /**
     * Return the Vector of the applets
     * @return Vector
     */
    public Vector getAppletVector() {
        return appVec;
    }

    /**
     * Return an Enumeration of the applets
     * @return Enumeration
     */
    public Enumeration elements() {
        return appVec.elements();
    }

    /**
     * Produce a formatted display of the applet properties.
     * @return The formatted applets display.
     */
    public String toString() {
        StringBuffer sb = new StringBuffer();
        //* the following message won't go into the .jtr file, so will leave it as is.
        sb.append("Applet Class\tAppletID\tParms"); //+ AppletProperties.toHeader()); 
        sb.append(EOL);
        
        int handle = 1;
        for (Enumeration e = appVec.elements(); e.hasMoreElements(); ) {
            
            AppletProperties appObj = (AppletProperties) e.nextElement();
            sb.append(handle++);
            sb.append(". ");
            sb.append(appObj);
            sb.append(EOL);
        }
        return sb.toString();
    }
    

    /**
     * This method converts the incoming string representing hex digits to
     * a byte array. The hex bytes can be separated by ','
     * @param hexString the incoming string to convert
     * @return byteArray equivalent of hex string
     */
     private static byte[] convertHexToBytes( String hexString, int maxLength) 
        throws IllegalArgumentException {

        byte[] bArray = new byte[maxLength];
        int buffIndex = 0;

        int digitNumber = 0;
        hexString = hexString.trim();
        log(I18n.getString("convert.hexstring", config_id, hexString));
        if (hexString.startsWith("0x") ) {
            hexString = hexString.substring(2);
        }
        
        char[] aChars = hexString.toCharArray();
        for ( int i=0; i<aChars.length; i++ ){
            // log("processing " + aChars[i] + " at index " + i);
            boolean isColon = (aChars[i]==':');
            if (isColon && aChars[i+1] == '0' && aChars[i+2] == 'x') {
                //log("found ':0x' at " + i);
                i += 2;
                continue;
            }
            
            boolean isComma = (aChars[i]==',');
            if (isComma || (digitNumber==2)) {
                if (++buffIndex==bArray.length) {
                    throw new IllegalArgumentException(I18n.getString("invalid.hexstring", hexString)); 
                }
                bArray[buffIndex] = 0;
                digitNumber = 0;
                if (isComma) continue;
            }

            int theDigit = Character.digit(aChars[i], 16);
            if (theDigit==-1) {
                    throw new IllegalArgumentException(I18n.getString("invalid.hexstring", hexString)); 
            }
            bArray[buffIndex] = (byte)((bArray[buffIndex] << 4)+ theDigit);
            digitNumber++;
        }
        byte[] returnArray = new byte[buffIndex+1];
        java.lang.System.arraycopy( bArray, 0, returnArray, 0, returnArray.length);
        return returnArray;
     }
    
  /**
   * Can be used to test this class from the command line.
   *
   * @param args    the command-line arguments
   *
   */
    public static void main (String[] args) {
        stdOut = new PrintWriter(System.out, true);
        stdErr = new PrintWriter(System.err, true);
        
        if (args.length == 0) {
            stdErr.println(I18n.getString("no.args.supply"));
            System.exit(1);
        }

        String m_configFileName = "";
        boolean m_verbose;
        
        for (int i = 0; i < args.length; ++i) {
            if (args[i].equals("-f")) {
                m_configFileName = args[++i];
            } else if (args[i].equals("-verbose") ||
                    args[i].equals("-v")) {
                m_verbose = true;
            }
        }
        
        config_id=I18n.getString("config.id");
        log(I18n.getString("config.filename", config_id, m_configFileName));
        ConfigFileReader cfr = null;    
        try {
            cfr = new ConfigFileReader(null, null);
            cfr.read(m_configFileName);
        } catch (Exception e) {
            log(I18n.getString("configfilereader.exception", config_id, e));
            System.exit(1);            
	}
        
        AppletProperties apa[] = cfr.getAppletPropertiesArray();
        
        for (int i = 0; i < apa.length; ++i) {
             log(I18n.getString("applet.properties", new Object[] {config_id, new Integer(i), 
                                                                   apa[i].toString(), apa[i].getPackageName(), 
								   apa[i].getPackageAID()}));
        }
    }
    

  /**
   * Write a message to the log.
   * @param msg the message to be written
   */
    static void log(String msg) {
        stdOut.println(msg);
    }

    public class Info {
        private String name;
        private AppletID packageID;
        private String version;
        private boolean exportMap;
        private Vector applets = new Vector();
        
        public Info(String name, AppletID packageID, String version,
                    boolean exportMap) {
            this.name = name;
            this.packageID = packageID;
            this.version = (version == null) ? null : version.intern();
            this.exportMap = exportMap;
        }

        public AppletID getPackageAID() {
            return this.packageID;
        }

        public String getName() {
            return this.name;
        }

        public void addApplet(AppletProperties applet) {
            this.applets.addElement(applet);
        }

        public AppletProperties[] getApplets() {
            AppletProperties[] retVal = new AppletProperties[applets.size()];
            return (AppletProperties[])this.applets.toArray(retVal);
        }

        public String getVersion() {
            return this.version;
        }

        public boolean isExportMap() {
            return this.exportMap;
        }
    }   
}
