/*
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.invoke;

import com.sun.javacard.cjck.userinterface.FatalException;
import com.sun.javacard.cjck.userinterface.FatalException.Scope;
import com.sun.tck.bvtool.terminal.StatefulCardTerminal;
import com.sun.javacard.cjck.I18n;
import com.sun.javacard.cjck.userinterface.BinaryToolService;
import javax.smartcardio.Card;
import javax.smartcardio.CardException;
import javax.smartcardio.CardTerminal;
import javax.smartcardio.TerminalFactory;

import com.sun.javacard.cjck.userinterface.CardService;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CardSystemException;
import com.sun.javacard.globalimpl.GPCardService;
import com.sun.tck.jc.javatest.gui.GuiUserInteraction;
import com.sun.tck.jc.javatest.ui.UserInteraction;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.Vector;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * This class is used to provide CardService instances based
 * on card terminals availability. If there's no card terminals
 * available at a moment, then client should wait till some other
 * service releases a terminal.
 *
 * @author Mikhail Smirnov
 */
public class CardServicesPool {

    public enum State {NON_INITIALIZED, INITIALIZED, READY, NO_TERMINALS, INITIALIZE_ERROR};

    public static final int TIMEOUT = 300;
    public static final int MAX_CONNECTION_ATTEMPTS = 3;

    private static int capacity = 0;
    private static State state = State.NON_INITIALIZED;
    private static String reason = null;

    private static final LinkedBlockingQueue<StatefulCardTerminal> terminalsQueue =
            new LinkedBlockingQueue<StatefulCardTerminal>();
    private static Vector<String> terminalLabels = new Vector<String>();
    private static List<StatefulCardTerminal> availableTerminals = new ArrayList<StatefulCardTerminal>();
    private static UserInteraction ui = GuiUserInteraction.getUserInteraction();
    
    static {
        initTerminals();
    }

    /*
     * This method instantiates terminals queue and fills it with available
     * card terminals
     */
    private static boolean initTerminals() {
        try {
            System.out.println("Initializing terminals:"); //FIXME: debug output
            TerminalFactory factory = TerminalFactory.getDefault();
            List<CardTerminal> terminalsList = factory.terminals().list();
            /*
            //FIXME: uncomment above lines and remove mock implementation below
            //<mock implementation>
            terminalsList = new ArrayList<CardTerminal>();
            terminalsList.add(new FakeTerminal("FakeTerminal #1"));
            terminalsList.add(new FakeTerminal("FakeTerminal #2"));
            terminalsList.add(new FakeTerminal("FakeTerminal #3"));
            terminalsList.add(new FakeTerminal("FakeTerminal #4"));
            //</mock implementation>
            */
            if (terminalsList == null || terminalsList.isEmpty()) {
                state = State.NO_TERMINALS;
            } else {
                capacity = terminalsList.size();
                int i = 1;
                for (CardTerminal cad: terminalsList) {
                    terminalLabels.add(String.valueOf(i++));
                    availableTerminals.add(new StatefulCardTerminal(cad));
                }
                state = State.INITIALIZED;
            }
        } catch (CardException ex) {
            state = State.INITIALIZE_ERROR;
            reason = ex.toString();
        }
        return true;
    }

    /**
     * Returns the number of available terminals
     */
    public static int getCapacity() {
        return capacity;
    }

    /**
     * This method provides initialized CardService instance bounded to a terminal from
     * the pool. If there's no terminals available at a moment, this method will wait
     * for <link>TIMEOUT</link> seconds.
     * 
     * @throws GPException
     */
    //FIXME: there shouldn't be realService parameter in final implementation
    public static BinaryToolService getCardService(CardService realService, String[] args, PrintWriter log, PrintWriter ref) throws CardSystemException {
        BinaryToolService cardService = null;
        int count = 0;
        while(count++ < capacity) {
            try {
                StatefulCardTerminal terminal = null;
                synchronized(terminalsQueue) {
                    terminal = terminalsQueue.poll(TIMEOUT, TimeUnit.SECONDS);
                    if (terminal == null) {
                        throw new CardSystemException(I18n.getString("pool.waiting.timeout"));
                    }
                    System.out.println("Terminal: " + terminal + " is requested"); //FIXME: debug output
                    cardService = new GPCardService();
                    cardService.setRealCardService((CJCKCardService)realService);
                    cardService.setCardTerminal(terminal);
                    cardService.init(args, log, ref);
                    checkCardServiceState(cardService);

                }
                if (cardService.getState() == StatefulCardTerminal.State.VERIFIED
                        || cardService.getState() == StatefulCardTerminal.State.RECOVERED) {
                    System.out.println("Terminal: " + terminal + " is taken"); //FIXME: debug output
                    break;
                } else {
                    cardService = null;
                    System.out.println("Terminal: " + terminal + " is unavailable"); //FIXME: debug output
                }
            } catch (InterruptedException ex) {
                throw new CardSystemException(I18n.getString("pool.waiting.interrupted", ex));
            }
        }
        if (cardService == null) {
            throw new CardSystemException("no.more.terminals.available");
        }
        return cardService;
    }

    /**
     * Releases terminal and puts it back to queue
     * @param terminal
     */
    public static void releaseTerminal(StatefulCardTerminal terminal) {
        System.out.println("Terminal: " + terminal + " is free");//FIXME: remove debug output
        terminalsQueue.offer(terminal);
    }

    /**
     * This method checks if a static initialization of CardServicesPool was successful
     * @throws GPException
     */
    public static void checkState() throws CardSystemException {
        if (state == State.READY) {
            return;
        } else if (state == State.INITIALIZED) {
            if (!identifyCardTerminals()) {
                throw new CardSystemException(I18n.getString("pool.not.identified"));
            } else {
                state = State.READY;
            }
        } else if (state == State.NO_TERMINALS) {
            throw new CardSystemException(I18n.getString("pool.no.terminals"));
        } else if (state == State.INITIALIZE_ERROR) {
            throw new CardSystemException(I18n.getString("pool.initialization.error", reason));
        } else {
            throw new CardSystemException(I18n.getString("pool.not.initialized"));
        }
    }

    /**
     * This method is called when a terminal that is associated with this
     * cardService instance contains a card that should be replaced. If user
     * doesn't want to replace a card we continue using other terminals
     *
     * @param cardService
     * @throws GPException
     */
    public static synchronized void requireReplacement(BinaryToolService cardService) throws CardSystemException {
        System.out.println("Card replacement required for card service: " + cardService);//FIXME: debug output
        StatefulCardTerminal terminal = cardService.getCardTerminal();
        if (!ui.askForCardReplacement(terminal)) {
            replaceTerminal(terminal, cardService);
        } else {
            if (checkConnection(terminal)) {
                setCardID(terminal);
            } else {
                replaceTerminal(terminal, cardService);
            }
        }
        cardService.setState(StatefulCardTerminal.State.CREATED);
        checkCardServiceState(cardService);
    }



    private static void replaceTerminal(StatefulCardTerminal terminal, BinaryToolService cardService) throws CardSystemException {
        terminal.setState(StatefulCardTerminal.State.INVALID);
        capacity--;
        if (capacity == 0) {
            throw new CardSystemException("no.more.terminals.available");
        }
        try {
            StatefulCardTerminal anotherTerminal = terminalsQueue.poll(TIMEOUT, TimeUnit.SECONDS);
            if (anotherTerminal == null) {
                throw new FatalException(Scope.CAD_SYSTEM, "no.more.terminals.available");
            }
            cardService.setCardTerminal(anotherTerminal);
        } catch (InterruptedException ex) {
            throw new CardSystemException(I18n.getString("pool.waiting.interrupted", ex));
        }
    }

    private static void checkCardServiceState(BinaryToolService cardService) throws CardSystemException {
        if (cardService.getState() == StatefulCardTerminal.State.CREATED) {
            if (cardService.initializeCardService() && cardService.validateCardService()){
                cardService.setState(StatefulCardTerminal.State.VERIFIED);
            } else {
                cardService.setState(StatefulCardTerminal.State.REPLACEMENT_REQUIRED);
                requireReplacement(cardService);
            }
        } else if (cardService.getState() == StatefulCardTerminal.State.VERIFIED) {
            try {
                StatefulCardTerminal terminal = cardService.getCardTerminal();
                if (!terminal.checkCardPresent()) {
                    requireReplacement(cardService);
                    String cardID = ui.showCardIDSelectDialog(terminal.getLabel());
                    if (!checkCardID(cardID)) {
                        throw new CardSystemException(I18n.getString("card.id.incorrect", cardID));
                    }
                    terminal.setInsertedCardID(cardID);
                }
            } catch (CardSystemException ex) {
                cardService.setState(StatefulCardTerminal.State.REPLACEMENT_REQUIRED);
                requireReplacement(cardService);
            }
        }
    }

    /**
     * This method checks connection between terminal and card, asks for card's ID
     * and after these checks adds terminal to the queue
     * @param terminal
     * @throws CardSystemException
     */
    private static void addTerminalToQueue(StatefulCardTerminal terminal) throws CardSystemException {
        if (checkConnection(terminal)) {
            setCardID(terminal);
            terminalsQueue.offer(terminal);
        } else {
            capacity--;
        }
    }

    /**
     * Asks user to set ID of the card in the terminal
     * @param terminal
     * @throws CardSystemException
     */
    private static void setCardID(StatefulCardTerminal terminal) throws CardSystemException {
        String cardID = ui.showCardIDSelectDialog(terminal.getLabel());
        if (!checkCardID(cardID)) {
            throw new CardSystemException(I18n.getString("card.id.incorrect", cardID));
        }
        terminal.setInsertedCardID(cardID);
    }

    public static List<StatefulCardTerminal> getAvailableTerminals() {
        return availableTerminals;
    }

    /**
     * This method creates threads for each terminal to wait for card insertion.
     * Then it asks user to insert cards one after another into terminals with
     * indicated labels. When a terminal detects card insertion event it is associated
     * with this label.
     *
     * @throws GPException
     */
    private static boolean identifyCardTerminals() throws CardSystemException {
        if (capacity == 1) {
            StatefulCardTerminal terminal = availableTerminals.get(0);

            String terminalLabel = terminalLabels.elementAt(0);
            try {
                if (!terminal.isCardPresent()) {
                    ui.askForCardInsertion(terminalLabel, false);
                }
            } catch (CardException ex) {
                throw new CardSystemException(I18n.getString("pool.not.initialized"));
            }
            terminal.setLabel(terminalLabel);
            addTerminalToQueue(terminal);
        } else {
            for (StatefulCardTerminal terminal : availableTerminals) {
                try {
                    //Before identification terminals should be empty
                    if (terminal.isCardPresent()) {
                        throw new CardSystemException(I18n.getString("pool.not.identified.nonempty"));
                    }
                } catch (CardException ex) {
                    throw new CardSystemException(I18n.getString("pool.initialization.error", ex.getMessage()));
                }
            }
            for (int i = 0; i < capacity; i++) {
                String terminalLabel = terminalLabels.remove(0);
                StatefulCardTerminal identifiedTerminal = ui.askForCardInsertion(terminalLabel, false);
                if (identifiedTerminal == null) {
                    break;
                } else {
                    identifiedTerminal.setLabel(terminalLabel);
                    addTerminalToQueue(identifiedTerminal);
                }
            }
        }
        capacity = terminalsQueue.size();
        if (capacity < 1) {
            state = State.INITIALIZE_ERROR;
            return false;
        }
        for (StatefulCardTerminal terminal : terminalsQueue) {
            if (terminal.getLabel() == null || terminal.getInsertedCard() == null) {
                state = State.INITIALIZE_ERROR;
                return false;
            }
        }
        return true;
    }

    /**
     * Checks connection between card and terminal with CardTerminal.connect() method
     * This method doesn't check other things like possibility to install or select some
     * applets.
     *
     * @param terminal
     * @return true is connection is OK, false otherwise.
     * @throws CardSystemException
     */
    private static boolean checkConnection(StatefulCardTerminal terminal) throws CardSystemException {
        int connectionAttempts = 0;
        boolean connectOK = false;
        while (!(connectOK = terminal.checkConnection()) && ++connectionAttempts < MAX_CONNECTION_ATTEMPTS) {
            if (ui.askForRetryAfterFailedConnection(terminal.getLabel())) {
                if (!ui.askForCardReplacement(terminal)) {
                    throw new CardSystemException(I18n.getString("card.reinsertion.failed"));
                }
            } else {
                return false;
            }
        }
        return connectOK;
    }

    private static boolean checkCardID(String cardID) {
        return cardID != null; //TODO: probably some other checks should be added
    }

}

/**
 *
 * Mock implementation of CardTerminal for testing Binary Tool framework without
 * real card terminals
 */
class FakeTerminal extends CardTerminal {

    private static int waitingTerminals = 0;
    private static synchronized int getWaitingTerminals() {
        return waitingTerminals++;
    }
    private static synchronized void decreaseWaitingTerminals() {
        waitingTerminals--;
    }

    private String name;
    private boolean isCardPresent = false;
    public FakeTerminal(String name) throws CardException {
        super();
        this.name = name;
    }

    @Override
    public Card connect(String protocol) throws CardException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public boolean isCardPresent() throws CardException {
        return isCardPresent;
    }

    @Override
    public boolean waitForCardAbsent(long timeout) throws CardException {
        return false;
    }

    @Override
    public boolean waitForCardPresent(long timeout) throws CardException {
        try {
            System.out.println("Waiting for a card to be inserted");
            isCardPresent = true;
            Random r = new Random();
            int waitSeconds = 2 + getWaitingTerminals();
            Thread.sleep(waitSeconds * 1000);
            decreaseWaitingTerminals();
        } catch (InterruptedException ex) {
            //ignore
        }
        return true;
    }

    @Override
    public String toString() {
        return "Fake Terminal: " + getName() + "[id: " + hashCode() + "]";
    }

}
