/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.invoke;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Date;
import java.util.Enumeration;
import java.util.Properties;

import com.sun.javacard.cjck.I18n;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CardService;
import com.sun.javacard.cjck.userinterface.CardServiceFactory;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javatest.Command;
import com.sun.javatest.Status;

/**
 * The cJCK's test driver.
 * It creates an instance of a CJCKCardService class to run a test.
 * When the test completes, it analyzes the output.
 * @see CJCKCardService
 * @version Version:%I% Date:%E%
 *
 * @author Eric Nellen (eric.nellen@eng.sun.com)
 * @author Maxim V. Sokolnikov
 */
public class CardProxyTest extends Command {

    static String VERSION = "2.2.1";
    static Date BUILD_DATE = new Date();
    static final String CONTACTED_SUPPORT = "contacted_support";
    static final String CONTACTLESS_SUPPORT = "contactless_support";
    static final String DUAL_INTERFACE_SUPPORT = "dual_interface_support";
    
    DelayedPrintWriter logWriter;
    DelayedPrintWriter refWriter;
    
    private static int instance;            // the instance number
    private int myInstance;
    private CardProxyArguments args;
    String log_id;
    
    ConfigFileReader configFile;
    CJCKCardService cardService;
    String runMode;
    private CJCKCardService contactedCardService = null;
    private CJCKCardService contactlessCardService = null;    
    /**
     * The default constructor.
     */
    public CardProxyTest() {
    }
    
    /**
     * The special constructor.
     * @param aCardService is an existing card service.
     */
    public CardProxyTest(CJCKCardService aCardService) {
        myInstance = ++instance;
        this.cardService = aCardService;
        log_id = I18n.getString("cardproxy.log.id", new Integer(myInstance));
    }
    
    
    /**
     * Can be used to run CardProxyTest from the command line.
     *
     * @param args  the command-line arguments
     *
     */
    public static void main(String[] args) {
        Status status;
        PrintWriter log = new PrintWriter(System.out, true);
        DelayedPrintWriter logWriter = new DelayedPrintWriter(log);
        DelayedPrintWriter refWriter = new DelayedPrintWriter(new PrintWriter(System.err));
        logWriter.println(I18n.getString("cardproxy.version.build", VERSION, BUILD_DATE));
        CardProxyTest test = new CardProxyTest();
        status = test.run(args, logWriter, refWriter);
        if (status.isPassed()) {
            logWriter.resetData();
            refWriter.resetData();
        } else {
            logWriter.flushData();
            refWriter.flushData();
        }
        log.println(I18n.getString("cardproxy.run.status", status));
        status.exit();  // return the status to JavaTest
    }
    
    
    /**
     * The method invoked by JavaTest if CardProxyTest is used as a JavaTest command.
     *
     * @param args  the command-line arguments supplied by JavaTest
     * @return Status a JavaTest status object
     *
     */
    public Status run(String[] arguments, PrintWriter log, PrintWriter ref) {
        // save the parameters
        this.logWriter = new DelayedPrintWriter(log);
        this.refWriter = new DelayedPrintWriter(ref);

        log_id = I18n.getString("cardproxy.log.id", new Integer(instance));
        log(I18n.getString("run.starting.msg", log_id, VERSION, BUILD_DATE));
        
        Status status = instantiateResources(arguments);
        if (!status.isPassed()) {
            return status;
        }
        
        log(I18n.getString("java.classpath", System.getProperty("java.class.path")));
        try {
            RunMode mode = createMode(runMode);
            if (contactedCardService != null
                    && contactlessCardService != null
                    && !runMode.equals("convert")
                    && args.isDuplicatedByBothIO()) {
                status = null;
                ref(I18n.getString("contacted.test.started"));                
                mode.init(logWriter, refWriter, contactedCardService);
                Status runStatus = mode.run(args, configFile);
                ref(I18n.getString("contacted.test.status", runStatus));
                ref(I18n.getString("contacted.test.finished"));
                if (!runStatus.isPassed()) {
                    status = runStatus;
                }
                ref(I18n.getString("contactless.test.started"));
                mode.init(logWriter, refWriter, contactlessCardService);
                runStatus = mode.run(args, configFile);
                ref(I18n.getString("contactless.test.status", runStatus));
                ref(I18n.getString("contactless.test.finished"));
                if (status == null) {
                    status = runStatus;
                }
            } else {
                mode.init(logWriter, refWriter, cardService);
                status = mode.run(args, configFile);
            }
        } catch (CardProxyException e) {
            status = Status.failed(I18n.getString("cardservice.exception", e));
        } catch (Throwable t) {
            t.printStackTrace(this.logWriter);
            logWriter.flushData();
            refWriter.flushData();
            return Status.error(I18n.getString("unexpected.exception", t));
        } finally {
            finalizeTest();
        }
        if (!status.isPassed()) {
            logWriter.flushData();
            refWriter.flushData();
        } else {
            logWriter.resetData();
            refWriter.resetData();
        }
        return status;
    }
    
    protected RunMode createMode(String mode) throws CardProxyException {
        if (mode.equals("convert")) {
            return new ConvertMode();
        } else if (mode.equals("install")) {
            return new InstallMode();
        } else {
            throw new CardProxyException(I18n.getString("unknown.mode", mode));
        }
        
    }
    
    /**
     * creates URLClassLoader for a given path. The path is assumed as list of
     * the files or directories separated by OS dependent path separator.
     * @param path
     * @return
     */
    /*protected ClassLoader createPathClassLoader(final String path)  {
        ArrayList retVal = new ArrayList();
        log(I18n.getString("starting.create.classloader", log_id));
        for (StringTokenizer e = new StringTokenizer(path, File.pathSeparator);
        e.hasMoreTokens();) {
            File f = new File(e.nextToken());
            try {
                if (f.exists()) {
                    retVal.add(f.toURL());
                    log(f.toURL() + "");
                }
            } catch (Exception e1) {
            }
        }
        log(I18n.getString("ending.create.classloader", log_id));
        return new URLClassLoader((URL[])retVal.toArray(new URL[retVal.size()]),
                this.getClass().getClassLoader());
    }
    */
    
    /**
     * Instantiates and initializes CardServise and ConfigFileReader.
     * @param args
     * @return
     */
    protected Status instantiateResources(String args[]) {
        try {
            this.args = new CardProxyArguments(args);
            
            // instantiate the ConfigFileReader
            configFile = new ConfigFileReader(logWriter, refWriter);
            for (Enumeration e = this.args.getLibConfigFiles(); e.hasMoreElements();
            configFile.read((String)e.nextElement()));
            configFile.read(this.args.getConfigFile());
            runMode = this.args.getRunMode();
            // instantiate a CJCKCardService
            instantiateCardServices();
            return Status.passed("");
        } catch (CardProxyException e) {
            return Status.error(I18n.getString("cardservice.init.exception", e));
        } catch (IllegalArgumentException ex) {
            return Status.failed(I18n.getString("can.not.parse.arg", ex));
        } catch (IOException io) {
            return Status.error(I18n.getString("can.not.read.config", io));
        }
    }
    
    /**
     * Depending on what card services supported by implementation this method
     * instantiate card services implementing CJCKCardService interface for
     * contacted or contactless or both I/O. Each CJCKCardService instance is created
     * based on CardService implementation class name and supported I/Os from interview
     */
    protected void instantiateCardServices() throws CardProxyException {
        Properties argProps = args.getProperties();
        // Get supported I/O interfaces
        boolean contactedSupport = argProps.getProperty(CONTACTED_SUPPORT, "true").equals("true");
        boolean contactlessSupport = argProps.getProperty(CONTACTLESS_SUPPORT, "false").equals("true");
        boolean dualSupport = contactedSupport && contactlessSupport &&
                argProps.getProperty(DUAL_INTERFACE_SUPPORT, "false").equals("true");
        int baseInterface;
        if (contactedSupport) {
            log(I18n.getString("contacted.init"));
            contactedCardService = new WrapperCardService(getBaseCardService(), CardService.CONTACTED_INTERFACE);
        }
        if (contactlessSupport) {
            if ((args.isDuplicatedByBothIO() && !runMode.equals("convert")) ||
                    !contactedSupport) {
                log(I18n.getString("contactless.init"));
                contactlessCardService = new WrapperCardService(getBaseCardService(), CardService.CONTACTLESS_INTERFACE, dualSupport);
            }
        }
        if (!contactedSupport && !contactlessSupport) {
            throw new CardProxyException(I18n.getString("io.not.supported"));
        }
        cardService = (contactedCardService != null) ? contactedCardService : contactlessCardService;
    }
    
    /**
     * This method creates and initializes CardService implementation that will be
     * used in CJCKCardService instance.
     */
    protected CardService getBaseCardService() throws CardProxyException {
        // instantiate a CJCKCardService
        CardService baseService;
        String name = args.getCardServiceClass();
        String path = args.getClassPath();
        log(I18n.getString("instantiate.cjckcardservice", log_id, name, path));
        //FIXME: mock implementation
        /*CardService realService = CardServiceFactory.getCardService(name,
                    getClassLoader());*/
        ClassLoader loader = getClassLoader();
        if (loader == null) {
            loader = args.getClassLoader();
        }
        baseService = CardServiceFactory.getCardService(name, loader);
        log(I18n.getString("init.cjckcardservice", log_id, name));
        baseService.init(args.getCardServiceArgs(), logWriter,
                refWriter);
        // initialize the Proxy
        baseService.startTest(args.getOutputDir(), args.getNuberOfReset());
        log(I18n.getString("cardservice.workdir.scriptcount", log_id,
                args.getOutputDir(), new Integer(args.getScriptFileCount())));
        return baseService;
    }
    
    /**
     * Power down the proxy.
     * New in 2.1.1
     * @return status
     */
    Status finalizeTest() {
        try {
            // powerDown the Proxy
            log(I18n.getString("call.powerdown", log_id));
            if (cardService != null) {
                cardService.powerDown();
                cardService.stopTest();
            }
            return Status.passed("");
        } catch (CardProxyException e) {
            return Status.failed(I18n.getString("powerdown.got.exception", e));
        }
    }
    
    /**
     * Write a message to the log.
     * @param msg the message to be written
     */
    void log(String msg) {
        logWriter.println(msg);
    }
    
    void ref(String msg) {
        log(msg);
        refWriter.println(msg);
    }
};
