/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck;

import java.io.PrintWriter;
import java.util.Set;
import com.sun.javatest.Script;
import com.sun.javatest.Status;
import com.sun.javatest.Test;
import com.sun.javatest.TestDescription;
import com.sun.javatest.TestEnvironment;
import com.sun.javatest.TestResult;
import com.sun.javatest.util.StringArray;

import com.sun.jck.lib.Expr;
import com.sun.javacard.cjck.invoke.CardTest;
import com.sun.javatest.util.I18NResourceBundle;
import com.sun.tck.jc.javatest.QueueTestRunner;
import com.sun.tck.jc.javatest.RerunnableTestDescription;

/**
 * A Script to convert/install a standard test.
 * The StdTestScript.java have been adapted for JC TCK and renamed.
 *
 * @author Jonathan J Gibbons
 * @author Maxim V. Sokolnikov
 * @version %W% %E%
 */
public class JCScript extends Script {
    private static final String SCRIPT_ARGS_KEY = "ScriptArgs";
    private static final String SIGNATURE_TEST_KEYWORD = "signaturetest";

    private boolean install = false;
    private static final I18NResourceBundle i18n = I18NResourceBundle.getBundleForClass(Script.class);
    /**
     * Convert and/or install a test.
     * The sources will be converted if `-convert' is given;
     * the test will be installed if
     * `-install' is given. Either one or both of install the convert options
     * and the install option must be given.
     *
     * By default, the compilation and execution are both expected to succeed.
     * However, if `-expectFail' is given as well as `-install', the execution
     * of the test will be expected to fail, in which case the test as a whole
     * will be deemed to have passed or to need its output to be checked
     * depending on whether `-checkErrors' is also given.)
     * If `-expectFail' is given but the test is simply converted, and not installed,
     * the expectation of failure applies to the compilation.
     *
     * @see javasoft.sqe.javatest.Script#compileTogether
     * @see javasoft.sqe.javatest.Script#compileIndividually
     * @see javasoft.sqe.javatest.Script#execute
     *
     */
    public Status run(String[] arguments, TestDescription td, TestEnvironment env) {
        boolean checkErrors = false;
        boolean expectFail = false;
        boolean convert = false;
        boolean convertIndividually = false;
        String args[] = null;
        try {
            args = env.lookup(SCRIPT_ARGS_KEY);
            args = (args == null) ? new String[0] : args;
        } catch (TestEnvironment.Fault e) {
            args = new String[0];
            e.printStackTrace();
        }
        
        if (!isSelected(td, env)) {
            return Status.passed(I18n.getString("test.have.not.been.selected"));
        }
        for (int i = 0; i < args.length; i++) {
            String arg = args[i];
            if (arg.equals("-convert") || arg.equals("-convertOnly") )
                convert = true;
            else if (arg.equals("-install"))
                install = true;
            else if (arg.equals("-expectFail"))
                expectFail = true;
            else if (arg.equals("-checkErrors"))
                checkErrors = true;
            else
                return Status.failed(I18n.getString("bad.arg.for.script.1", arg));
        }

        Set keys = td.getKeywordTable();
        negativeKey = keys.contains("negative");
        if (keys.contains(SIGNATURE_TEST_KEYWORD)) {
            return runSignatureTest(arguments, td, env);
        }
        if (convert) {
            Status convertStatus;
            String executeClass = td.getParameter("executeClass");

            convertStatus = convert(executeClass, td.getParameter("executeArgs"));
            if (!(convertStatus.isPassed() || convertStatus.isFailed())) {
                return convertStatus;
            }
            if (negativeKey) {
                switch (convertStatus.getType()) {
                case Status.PASSED:
                    return Status.failed(I18n.getString("converter.passed.negative.test"));
                case Status.FAILED:
                    return Status.passed(I18n.getString("converter.failed.as.expected"));
                }
            }
            
            if (!hasInstallKeyword(keys))
                return convertStatus;
 
            // if we're not going to install the test, this is the end of the task
            if (!install) {
                if (expectFail) {
                    if (convertStatus.getType() == Status.FAILED) {
                        if (checkErrors)
                            return error_compFailUnexp.augment(convertStatus);
                        else
                            return pass_compFailExp.augment(convertStatus);
                    } else
                        return fail_compSuccUnexp.augment(convertStatus);
                } else
                    return convertStatus;
            } else {
                // if we want to install the test, but the compilation failed, we can't go on
                if (convertStatus.getType() == Status.FAILED)
                    return fail_compFailUnexp.augment(convertStatus);
            }
        }

        if (install) {
            String executeClass = td.getParameter("executeClass");
            if (executeClass == null)
                return error_noExecuteClass;

            Status executeStatus = install(executeClass, td.getParameter("executeArgs"));

            if (expectFail) {
                if (executeStatus.getType() == Status.FAILED) {
                        return pass_execFailExp.augment(executeStatus);
                } else {
                    return fail_execSuccUnexp.augment(executeStatus);
                }
            } else
                return executeStatus;
        }

        return error_noActionSpecified;
    }

    private Status runSignatureTest(String[] arguments, 
                                    TestDescription td, TestEnvironment env) {
        String name = td.getParameter("executeClass");
        String args = td.getParameter("executeArgs");
        PrintWriter out = null;
        PrintWriter ref = null;
        TestResult.Section section = getTestResult().createSection("execute");
        try {
            String[] list = env.resolve(args);
            section.getMessageWriter().println(I18n.getString("script.command", 
                                                              new Object[] {name, StringArray.join(list) }));
            out = section.createOutput(cmdOut1Name);
            ref = section.createOutput(cmdOut2Name);
            Test sigTest = (Test)Class.forName(name).newInstance();
            Status retVal = sigTest.run(list, out, ref);
            section.setStatus(retVal);
            return retVal;
        } catch (Exception e) {
            e.printStackTrace(trOut);
            Status retVal = Status.error("Unexpected exception:" + e);
            section.setStatus(retVal);
            return retVal;           
        } finally {
            if (out != null) {
                out.close();
            }
            if (ref != null) {
                ref.close();
            }
	    }
    }

    public Status convert(String executeClass, String executeArgs) {
        return execute(DEFAULT_CONVERT_COMMAND, executeClass, executeArgs);
    } 
    public Status install(String executeClass, String executeArgs) {
        return execute(DEFAULT_INSTALL_COMMAND, executeClass, executeArgs);
    }

    @Override
    protected Status invokeCommand(String key) {

        TestResult.Section section;
	Status s = null;

	try {
	    String[] command = env.lookup("command." + key);

	    if (command.length == 0)
		return Status.error(i18n.getString("script.noCommand",
						   new Object[] { env.getName(), key }));

	    String className = command[0];
	    String[] args = new String[command.length - 1];
	    System.arraycopy(command, 1, args, 0, args.length);
            TestResult testResult = getTestResult();
            section = testResult.createSection(key);

	    section.getMessageWriter().println(i18n.getString("script.command",
							      new Object[] {className, StringArray.join(args) }));

	    PrintWriter out1 = null;
	    PrintWriter out2 = null;
	    try {
                out1 = section.createOutput(cmdOut1Name);
                out2 = section.createOutput(cmdOut2Name);
		//s = invokeClass(className, args, out1, out2);
                QueueTestRunner testRunner = QueueTestRunner.getTestRunner();
                testRunner.testStarted();
                CardTest test = new CardTest(((RerunnableTestDescription)td).isRerun(), this);
                System.out.println("Invoking CardTest.run() for " + td.getRootRelativeURL());//FIXME: debug output
                s = test.run(args, out1, out2);
                if (s.isFailed() || s.isError()) {
                    if (test.getState() == CardTest.State.WAITING_FOR_RERUN) {
                        testRunner.returnToQueue(td);
                    }
                }
                testRunner.testCompleted();
                out1.close();
                out2.close();
	    }
	    finally {
		if (out2 != null)  out2.close();
		if (out1 != null)  out1.close();
	    }

            section.setStatus(s);
	    return s;
	}
	catch (TestEnvironment.Fault e) {
	    return Status.error(i18n.getString("script.badCommand",
					       new Object[] { env.getName(), key }));
	}
    }



    private static final String DEFAULT_CONVERT_COMMAND = "convert";
    private static final String DEFAULT_INSTALL_COMMAND = "install";

    // Keys are set from test description keywords
    //
    
    protected boolean negativeKey;

    public boolean isSelected(TestDescription td, TestEnvironment env) {
        try {
            String s = td.getParameter("selectIf");
            if ((s == null) || (s.length() == 0)) {
                return true;
            }
            Expr e = Expr.parse(s);
            return e.evalBoolean(env);
        } catch (Exception e) {
            throw new IllegalArgumentException(e.toString());
        }
    }

    public boolean hasInstallKeyword(Set keys) {
        return (keys.contains("install") || keys.contains("runtime"));
    }

    public void resetTimeout() {
        setAlarm(0);
        int delayMs = getTestTimeout() * 1000;
        setAlarm(delayMs);
    }

    public void cancelTimeout() {
        setAlarm(0);
    }
}
