/**
 * %W% %E%
 * 
 * @Copyright
 */
package com.sun.bvt.launcher;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

/**
 * @author Mikhail Smirnov
 */
public class Main {

    private final static long COLLECTOR_SLEEP_TIME = 200;
    private final static long CHECKER_SLEEP_TIME = 100;
    private final static long SLEEP_TIME = 2000;
    private final static int STATE_ERROR = -1;
    private final static String PROC_NAME = "Binary Verification Tool";
    private final static String JAR_NAME = "bvt-launcher.jar";

    private static Descriptor descriptor;
    private static Collector collector = new Collector();
    private static String cmdArgs = "";
    private static String jvmArgs = "";
    private static String tckHome = null;

    public static void main(String[] args) {
        parseArgs(args);

        String javaHome = System.getProperty("java.home");
        if (javaHome == null) {
            javaHome = System.getenv("JAVA_HOME");
        }
        if (tckHome == null) {
            tckHome = System.getenv("TCK_HOME");
        }
        String classPath = System.getProperty("java.class.path");
        String[] elements = classPath.split(File.pathSeparator);
        String jarLocation = null;
        for (String element : elements) {
            int endIdx = element.indexOf(JAR_NAME);
            if (endIdx == -1) {
                continue;
            } else if (endIdx == 0) {
                jarLocation = "./";
            } else {
                jarLocation = element.substring(0, endIdx);
            }
        }
        if (jarLocation == null) {
            jarLocation = tckHome + "/lib/";
        }

        String cmd = javaHome + "\\bin\\java " + jvmArgs
                + " -Xbootclasspath/p:" + jarLocation
                + "bvt-scio.jar "
                + " -Djavatest.security.noSecurityManager=true"
                + " -jar " + jarLocation + "javatest.jar "
                + cmdArgs;
        int status = execute(PROC_NAME, cmd, null);
        System.exit(status);
    }

    private static void parseArgs(String[] args) {
        for (int i = 0; i < args.length; i++) {
            if ("-testsuite".equals(args[i].toLowerCase())) {
                tckHome = args[i + 1];
            }
            if (args[i].startsWith("-D") || args[i].startsWith("-X")) {
                jvmArgs += args[i] + " ";
            } else {
                cmdArgs += args[i] + " ";
            }
        }
    }

    private static int execute(String processName, String cmd, String env[]) {
        try {
            descriptor = new Descriptor(processName, cmd, env);
            if (!collector.isAlive()) {
                collector.start();
            }
        } finally {
            return waitUntilDone(descriptor);
        }
    }

    private static int waitUntilDone(Descriptor current) {
        if (current == null || current.process == null) {
            return -1;
        }
        try {
            int status = current.close(false);
            collector.finish();
            return status;
        } catch (RuntimeException ex) {
            return -1;
        }
    }

    private static String readAll(InputStream in) {
        if (in == null) {
            return "";
        }
        int pos = 0;
        byte[] buff = {};
        try {
            int length = in.available();
            if (length == 0) {
                return "";
            }
            buff = new byte[in.available()];
            int bytesRead = 0;
            while ((bytesRead = in.read(buff, pos, buff.length - pos)) > 0 && pos < buff.length) {
                pos += bytesRead;
            }
        } catch (Exception e) {
//            e.printStackTrace();
        }
        return new String(buff, 0, pos);
    }

    private static void readBuffers(Descriptor item) {
        if ((item == null) || !item.isActive) {
            return;
        }
        if (item.err != null) {
            String errData = readAll(item.err);
            if (errData != null) {
                System.err.print(errData);
            }
        }
        if (item.out != null) {
            String data = readAll(item.out);
            if (data != null) {
                System.out.print(data);
            }
        }
    }
    
    private static class Collector extends Thread {

        private boolean stopped = false;

        public Collector() {
            super("CollectorThread");
        }

        public void run() {
            try {
                while (!stopped) {
                    readBuffers(descriptor);
                    Thread.sleep(COLLECTOR_SLEEP_TIME);
                }
            } catch (InterruptedException e) {
                return;
            }
        }

        private void finish() {
            stopped = true;
        }
    }

    private static class Descriptor {

        private boolean isActive = true;
        private String name;
        private Process process;
        private InputStream out;
        private InputStream err;
        private int state;

        public Descriptor(String name, String cmd, String env[]) {
            this.name = name;
            while (true) {
                try {
                    this.process = Runtime.getRuntime().exec(cmd, env, null);
                    this.err = this.process.getErrorStream();
                    this.out = this.process.getInputStream();
                    //We need to check whether process successfully started
                    //if not log error and throw exception
                    checkProcess(CHECKER_SLEEP_TIME);
                    return;
                } catch (IOException e) {
                    e.printStackTrace();
                    try {
                        Runtime run = Runtime.getRuntime();
                        run.gc();
                        run.runFinalization();
                        Thread.sleep(SLEEP_TIME);
                    } catch (InterruptedException ie) {
                        if (this.process != null) {
                            close(true);
                        }
                        state = STATE_ERROR;
                        return;
                    }
                }
            }
        }

        /**
         * Checks if the process still alive after given wait time.
         * @param waitTime - time in milliseconds to wait for process completion.
         * process is not alive
         */
        public void checkProcess(long waitTime) {
            ProcessChecker checker = new ProcessChecker(this.process);
            checker.start();
            try {
                Thread.sleep(waitTime);
            } catch (InterruptedException ex) {
            }
            if (!checker.isInterrupted()) {
                checker.interrupt();
            }
            if (!checker.isProcessAlive()) {
                String errMessage = readAll(err);
                int exitCode = process.exitValue();
                process.destroy();
                checker = null;
                close(true);
                state = STATE_ERROR;
                System.err.println("Process ends with status '" + exitCode + "'.\nError message: " + errMessage);
            } else {
                checker = null;
                return;
            }
        }

        private int close(boolean isDestroy) {
            if (!isActive) {
                return state;
            }
            try {
                if (isDestroy) {
                    state = this.process.exitValue();
                } else {
                    state = this.process.waitFor();
                }
                return state;
            } catch (InterruptedException ex) {
                throw new RuntimeException(name + "  have been interrupted");
            } catch (IllegalThreadStateException e) {
                state = STATE_ERROR;
                return state;
            } finally {
                this.process.destroy();
                synchronized (this) {
                    readBuffers(this);
                    isActive = false;
                }
            }
        }

        /**
         * Thread implementation that checks if a process is alive
         */
        private static class ProcessChecker extends Thread {

            Process process;
            boolean isProcessAlive = true;

            public ProcessChecker(Process process) {
                super("ProcessCheckerThread");
                this.process = process;
            }

            @Override
            public void run() {
                try {
                    process.waitFor();
                    isProcessAlive = false;
                } catch (InterruptedException ex) {
                    return;
                }
            }

            public boolean isProcessAlive() {
                return isProcessAlive;
            }
        }
    }
}
